#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_utils.site module

This modules provides classes of elements which can be used as application or site "root"
objects.

Il also provides functions which are used to manage site's "generations", used to upgrade
objects while migrating from one version to another.
"""

from persistent.dict import PersistentDict
from pyramid.exceptions import NotFound
from pyramid.path import DottedNameResolver
from pyramid.security import ALL_PERMISSIONS, Allow, Everyone
from pyramid.threadlocal import get_current_registry
from pyramid_zodbconn import get_connection
from zope.component import hooks
from zope.component.interfaces import IPossibleSite
from zope.container.folder import Folder
from zope.interface import implementer
from zope.interface.interfaces import ObjectEvent
from zope.lifecycleevent import ObjectCreatedEvent
from zope.site.site import LocalSiteManager, SiteManagerContainer
from zope.traversing.interfaces import ITraversable

from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.interfaces import PUBLIC_PERMISSION, PYAMS_APPLICATION_DEFAULT_NAME, \
    PYAMS_APPLICATION_FACTORY_KEY, PYAMS_APPLICATION_SETTINGS_KEY
from pyams_utils.interfaces.site import IConfigurationManager, INewLocalSiteCreatedEvent, \
    ISiteGenerations, ISiteRoot, ISiteRootFactory, ISiteUpgradeEvent, SITE_GENERATIONS_KEY
from pyams_utils.registry import get_utilities_for, query_utility


__docformat__ = 'restructuredtext'


@implementer(ISiteRoot, IConfigurationManager)
class BaseSiteRoot(Folder, SiteManagerContainer):
    """Default site root

    A site root can be used as base application root in your ZODB.
    It's also site root responsibility to manage your local site manager.

    BaseSiteRoot defines a basic ACL which gives all permissions to system administrator,
    and 'public' permission to everyone. But this ACL is generally overriden in subclasses
    which also inherit from :py:class:`ProtectedObject <pyams_security.security.ProtectedObject>`.
    """

    __acl__ = [(Allow, 'system:admin', ALL_PERMISSIONS),
               (Allow, Everyone, {PUBLIC_PERMISSION})]

    config_klass = None


@adapter_config(name='etc', context=ISiteRoot, provides=ITraversable)
class SiteRootEtcTraverser(ContextAdapter):
    """Site root ++etc++ namespace traverser

    Gives access to local site manager from */++etc++site* URL
    """

    def traverse(self, name, furtherpath=None):  # pylint: disable=unused-argument
        """Traverse to site manager;
        see :py:class:`ITraversable <zope.traversing.interfaces.ITraversable>`"""
        if name == 'site':
            return self.context.getSiteManager()
        raise NotFound


@implementer(INewLocalSiteCreatedEvent)
class NewLocalSiteCreatedEvent(ObjectEvent):
    """New local site creation event"""


def site_factory(request):
    """Application site factory

    On application startup, this factory checks configuration to get application name and
    load it from the ZODB; if the application can't be found, configuration is scanned to
    get application factory, create a new one and create a local site manager.
    """
    conn = get_connection(request)
    root = conn.root()
    application_key = request.registry.settings.get(PYAMS_APPLICATION_SETTINGS_KEY,
                                                    PYAMS_APPLICATION_DEFAULT_NAME)
    application = root.get(application_key)
    if application is None:
        factory = request.registry.settings.get(PYAMS_APPLICATION_FACTORY_KEY)
        if factory:
            resolver = DottedNameResolver()
            factory = resolver.maybe_resolve(factory)
        else:
            factory = request.registry.queryUtility(ISiteRootFactory, default=BaseSiteRoot)
        application = root[application_key] = factory()
        if IPossibleSite.providedBy(application):
            lsm = LocalSiteManager(application, default_folder=False)
            application.setSiteManager(lsm)
        try:
            # if some components require a valid and complete registry
            # with all registered utilities, they can subscribe to
            # INewLocalSiteCreatedEvent event interface
            hooks.setSite(application)
            get_current_registry().notify(NewLocalSiteCreatedEvent(application))
        finally:
            hooks.setSite(None)
        import transaction  # pylint: disable=import-outside-toplevel
        transaction.commit()
    return application


@implementer(ISiteUpgradeEvent)
class SiteUpgradeEvent(ObjectEvent):
    """Site upgrade request event"""


def site_upgrade(request):
    """Upgrade site when needed

    This function is executed by *pyams_upgrade* console script.
    Site generations are registered named utilities providing
    :py:class:`ISiteGenerations <pyams_utils.interfaces.site.ISiteGenerations>` interface.

    Current site generations are stored into annotations for each generation adapter.
    """
    application = site_factory(request)
    if application is not None:
        try:
            hooks.setSite(application)
            generations = get_annotation_adapter(application, SITE_GENERATIONS_KEY, PersistentDict,
                                                 notify=False, locate=False)
            for name, utility in sorted(get_utilities_for(ISiteGenerations),
                                        key=lambda x: x[1].order):
                if not name:
                    name = '.'.join((utility.__module__, utility.__class__.__name__))
                current = generations.get(name)
                if not current:
                    print("Upgrading {0} to generation {1}...".format(name, utility.generation))
                elif current < utility.generation:
                    print("Upgrading {0} from generation {1} to {2}...".format(name, current,
                                                                               utility.generation))
                utility.evolve(application, current)
                generations[name] = utility.generation
        finally:
            hooks.setSite(None)
        import transaction  # pylint: disable=import-outside-toplevel
        transaction.commit()
    return application


def check_required_utilities(site, utilities):
    """Utility function to check for required utilities

    :param ISite site: the site manager into which configuration may be checked
    :param tuple utilities: each element of the tuple is another tuple made of the utility
        interface, the utility registration name, the utility factory and the object name when
        creating the utility, as in:

    .. code-block:: python

        REQUIRED_UTILITIES = ((ISecurityManager, '', SecurityManager, 'Security manager'),
                              (IPrincipalAnnotationUtility, '', PrincipalAnnotationUtility,
                               'User profiles'))
    """
    registry = get_current_registry()
    for interface, name, factory, default_id in utilities:
        utility = query_utility(interface, name=name)
        if utility is None:
            lsm = site.getSiteManager()
            if default_id in lsm:
                continue
            utility = factory()
            registry.notify(ObjectCreatedEvent(utility))
            lsm[default_id] = utility
            lsm.registerUtility(utility, interface, name=name)
