#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_utils.schema module

This module is used to define custom schema fields
"""

import re
import string

from persistent.list import PersistentList as PersistentListType
from persistent.mapping import PersistentMapping
from zope.interface import implementer, Interface
from zope.schema import Decimal, Dict, List, Password, Text, TextLine, Tuple, ValidationError
from zope.schema.interfaces import IDecimal, IDict, IList, IPassword, IText, ITextLine, ITuple


__docformat__ = 'restructuredtext'

from pyams_utils import _


#
# Persistent list field
#

class IPersistentList(IList):
    """Persistent list field marker interface"""


@implementer(IPersistentList)
class PersistentList(List):
    """Persistent list field"""

    _type = PersistentListType


#
# Persistent mapping field
#

class IPersistentDict(IDict):
    """Persistent mapping field marker interface"""


@implementer(IPersistentDict)
class PersistentDict(Dict):
    """Persistent mapping field"""

    _type = PersistentMapping


#
# Encoded password field
#

class IEncodedPassword(IPassword):
    """Encoded password field interface"""


@implementer(IEncodedPassword)
class EncodedPassword(Password):
    """Encoded password field"""

    _type = None

    def fromUnicode(self, str):  # pylint: disable=redefined-builtin
        return str

    def constraint(self, value):  # pylint: disable=method-hidden
        return True


#
# HTML field
#

class IHTMLField(IText):
    """HTML field interface"""


@implementer(IHTMLField)
class HTMLField(Text):
    """HTML field"""


#
# JSON dict value field
#

class IJSONDictField(IDict):
    """JSON dict value field interface"""


class IJSONDictFieldsGetter(Interface):
    """Adapter interface used to get JSON value fields list"""

    def get_fields(self, data):
        """Returns an iterator made of tuples

        Each tuple may ocntain field name, field label and field value
        """


@implementer(IJSONDictField)
class JSONDictField(Dict):
    """JSON dict value field"""

    def __init__(self, key_type=None, value_type=None, **kw):
        super(JSONDictField, self).__init__(key_type=TextLine(),
                                            value_type=TextLine(),
                                            **kw)


#
# Color field
#

class IColorField(ITextLine):
    """Marker interface for color fields"""


@implementer(IColorField)
class ColorField(TextLine):
    """Color field"""

    def __init__(self, *args, **kw):
        super(ColorField, self).__init__(max_length=6, *args, **kw)

    def _validate(self, value):
        if len(value) not in (3, 6):
            raise ValidationError(_("Color length must be 3 or 6 characters"))
        for val in value:
            if val not in string.hexdigits:
                raise ValidationError(_("Color value must contain only valid hexadecimal color "
                                        "codes (numbers or letters between 'A' end 'F')"))
        super(ColorField, self)._validate(value)


#
# Pointed decimal field
#

class IDottedDecimalField(IDecimal):
    """Marker interface for dotted decimal fields"""


@implementer(IDottedDecimalField)
class DottedDecimalField(Decimal):
    """Dotted decimal field"""


#
# Dates range field
#

class IDatesRangeField(ITuple):
    """Marker interface for dates range fields"""


@implementer(IDatesRangeField)
class DatesRangeField(Tuple):
    """Dates range field"""

    def __init__(self, value_type=None, unique=False, **kw):
        super(DatesRangeField, self).__init__(value_type=None, unique=False,
                                              min_length=2, max_length=2, **kw)


#
# TextLine list field
#

class ITextLineListField(IList):
    """Marker interface for textline list field"""


@implementer(ITextLineListField)
class TextLineListField(List):
    """TextLine list field"""

    def __init__(self, value_type=None, unique=False, **kw):
        super(TextLineListField, self).__init__(value_type=TextLine(), unique=True, **kw)


#
# Mail address field
#

class IMailAddressField(ITextLine):
    """Marker interface for mail address field"""


EMAIL_REGEX = re.compile(r"^[^ @]+@[^ @]+\.[^ @]+$")


class InvalidEmail(ValidationError):
    """Invalid email validation error"""

    __doc__ = _("Email address must be entered as « name@domain.name », without '<' and '>' "
                "characters")


@implementer(IMailAddressField)
class MailAddressField(TextLine):
    """Mail address field"""

    def _validate(self, value):
        super(MailAddressField, self)._validate(value)
        if not EMAIL_REGEX.match(value):
            raise InvalidEmail(value)


#
# Multiple addresses field
#

class IMailAddressesListField(ITextLine):
    """Marker interface for multiple mail addresses list"""


class InvalidEmailsList(ValidationError):
    """Invalid mail addresses list validation error"""

    __doc__ = _("Email addresses must be entered as « name@domain.name », without '<' and '>' "
                "characters, and separated by semicolons")


@implementer(IMailAddressesListField)
class MailAddressesListField(TextLine):
    """Mail addresses list field"""

    def _validate(self, value):
        super(MailAddressesListField, self)._validate(value)
        for address in value.split(';'):
            if not EMAIL_REGEX.match(address.strip()):
                raise InvalidEmailsList(value)
