#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from cgi import FieldStorage

import enum
from colander import Date, Enum, Mapping, MappingSchema, SchemaNode, SequenceSchema, String, Tuple, TupleSchema, drop, \
    null
from cornice_swagger import CorniceSwagger
from cornice_swagger.converters.schema import ArrayTypeConverter, ObjectTypeConverter, StringTypeConverter, \
    TypeConverter
from pyramid.httpexceptions import HTTPNotFound, HTTPUnauthorized, HTTPForbidden, HTTPBadRequest, HTTPServiceUnavailable


__docformat__ = 'restructuredtext'


#
# Colander schemas and converters
#

class STATUS(enum.Enum):
    """Base response status enumeration"""
    SUCCESS = 'success'
    ERROR = 'error'


class BaseStatusSchema(MappingSchema):
    """Base status schema"""
    status = SchemaNode(Enum(STATUS),
                        description="Response status",
                        missing=drop)


class BaseResponseSchema(BaseStatusSchema):
    """Base response schema"""
    message = SchemaNode(String(),
                         description="Error or status message",
                         missing=drop)


class BaseResponse(MappingSchema):
    """Base response"""
    body = BaseResponseSchema()


class EnumTypeConverter(TypeConverter):
    """Enum type converter"""

    type = 'string'

    def convert_type(self, schema_node):
        """Enum type converter"""
        converted = super().convert_type(schema_node)
        converted['enum'] = list(map(lambda x: x.value,
                                     schema_node.typ.values.values()))
        return converted


class StringListSchema(SequenceSchema):
    """Strings list schema field"""
    value = SchemaNode(String(),
                       description="String item value",
                       missing=drop)


class StringListTypeConverter(ArrayTypeConverter):
    """Strings list type converter"""


class PropertiesMapping(Mapping):
    """Properties schema"""

    name = 'properties'

    def serialize(self, node, appstruct):
        if appstruct is null:
            return {}
        return appstruct

    def deserialize(self, node, cstruct):
        return cstruct


class PropertiesMappingTypeConverter(ObjectTypeConverter):
    """Properties mapping type converter"""


class DateRangeSchema(TupleSchema):
    """Dates range schema type"""
    after = SchemaNode(Date(),
                       description="Range beginning date",
                       missing=null)
    before = SchemaNode(Date(),
                        description="Range ending date (excluded)",
                        missing=null)


class DateRangeTypeConverter(ArrayTypeConverter):
    """Date range type converter"""


class FileUploadType(String):
    """File upload type"""

    def deserialize(self, node, cstruct):
        """File upload deserializer"""
        if isinstance(cstruct, FieldStorage):
            return cstruct
        return super().deserialize(node, cstruct)


class FileUploadTypeConverter(StringTypeConverter):
    """File upload type converter"""


# update Cornice-Swagger types converters
CorniceSwagger.custom_type_converters.update({
    Enum: EnumTypeConverter,
    Tuple: ArrayTypeConverter,
    StringListSchema: StringListTypeConverter,
    PropertiesMapping: PropertiesMappingTypeConverter,
    DateRangeSchema: DateRangeTypeConverter,
    FileUploadType: FileUploadTypeConverter
})


rest_responses = {
    HTTPNotFound.code: BaseResponse(description=HTTPNotFound.title),
    HTTPUnauthorized.code: BaseResponse(description=HTTPUnauthorized.title),
    HTTPForbidden.code: BaseResponse(description=HTTPForbidden.title),
    HTTPBadRequest.code: BaseResponse(description=HTTPBadRequest.title),
    HTTPServiceUnavailable.code: BaseResponse(description=HTTPServiceUnavailable.title)
}


def http_error(request, error, message=None):
    """HTTP error response"""
    request.response.status_code = error.code
    return {
        'status': STATUS.ERROR.value,
        'message': '{0.title}: {1}'.format(error, message) if message else error.title
    }


def http_response(request, error, result, status=STATUS.SUCCESS):
    """HTTP server response"""
    request.response.status_code = error.code
    result['status'] = status.value
    return result
