#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import random
import time


class MaxRetriesException(Exception):
    """Maximum retries exception"""


def exponential_backoff(max_retries, base_delay):
    """Exponential backoff decorator

    Exponential backoff is an algorithm that uses feedback to
    decrease the rate of some process, in order to gradually
    find an acceptable rate.
    """

    def decorator(func):
        def wrapper(*args, **kwargs):
            retries = 0
            while retries < max_retries:
                try:
                    return func(*args, **kwargs)
                except Exception as e:
                    retries += 1
                    delay = (base_delay * 2 ** retries + random.uniform(0, 1))
                    time.sleep(delay)
            raise MaxRetriesException("Max retries reached, operation failed.")

        return wrapper

    return decorator
