#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_utils.dict module

This helper module only contains a single function which can be used to update an input
dictionary; if given value argument is a boolean 'true' value, given dictionary's key is created
or updated, otherwise dictionary is left unchanged.
"""

from html import escape

__docformat__ = 'restructuredtext'


def update_dict(input_dict, key, value):
    """Update given mapping if input value is a boolean 'true' value

    :param dict input_dict: input dictionary
    :param key: mapping key
    :param value: new value

    'False' values leave mapping unchanged::

    >>> from pyams_utils.dict import update_dict
    >>> mydict = {}
    >>> update_dict(mydict, 'key1', None)
    >>> mydict
    {}
    >>> update_dict(mydict, 'key1', '')
    >>> mydict
    {}
    >>> update_dict(mydict, 'key1', 0)
    >>> mydict
    {}

    'True' values modify the mapping::

    >>> update_dict(mydict, 'key1', 'value')
    >>> mydict
    {'key1': 'value'}
    >>> update_dict(mydict, 'key1', 'value2')
    >>> mydict
    {'key1': 'value2'}
    """
    if value:
        input_dict[key] = value


def escape_dict(value, in_place=False):
    """Escape all values from input dictionary
    
    :param dict value: input dictionary
    :param bool in_place: if set to True, input dictionary is modified in place
    
    :return: escaped dictionary

    >>> from pyams_utils.dict import escape_dict
    >>> input = {
    ...     'key1': '<value1 />',
    ...     'key2': {'key3': 'value2&value3'},
    ...     'key4': ['<value4 />'],
    ...     'key5': set(['<value5 />']),
    ...     'key6': 6
    ... }
    >>> escape_dict(input)
    {'key1': '&lt;value1 /&gt;', 'key2': {'key3': 'value2&amp;value3'}, 'key4': ['&lt;value4 /&gt;'], 'key5': {'&lt;value5 /&gt;'}, 'key6': 6}

    Initial value is not modified::
    >>> input
    {'key1': '<value1 />', 'key2': {'key3': 'value2&value3'}, 'key4': ['<value4 />'], 'key5': {'<value5 />'}, 'key6': 6}

    >>> escape_dict(input, in_place=True)
    {'key1': '&lt;value1 /&gt;', 'key2': {'key3': 'value2&amp;value3'}, 'key4': ['&lt;value4 /&gt;'], 'key5': {'&lt;value5 /&gt;'}, 'key6': 6}
    >>> input
    {'key1': '&lt;value1 /&gt;', 'key2': {'key3': 'value2&amp;value3'}, 'key4': ['&lt;value4 /&gt;'], 'key5': {'&lt;value5 /&gt;'}, 'key6': 6}
    """
    result = value if in_place else {}
    for key, val in value.items():
        if isinstance(val, str):
            result[key] = escape(val)
        elif isinstance(val, dict):
            result[key] = escape_dict(val)
        elif isinstance(val, list):
            result[key] = [
                escape_dict(item) if isinstance(item, dict) else escape(item)
                for item in val
            ]
        elif isinstance(val, set):
            result[key] = {
                escape_dict(item) if isinstance(item, dict) else escape(item)
                for item in val
            }
        else:
            result[key] = val
    return result
