#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_skin.interfaces.container import ITableWithActions
from pyams_skin.interfaces.viewlet import IToolbarViewletManager, IToolbarAction, IToolbarMenu, IToolbarMenuItem, \
    IContextActions, IToolbarAddingMenu, ITableItemColumnActionsMenu, IWidgetTitleViewletManager, IToolbarActionItem
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces.tales import ITALESExtension
from zope.contentprovider.interfaces import IContentProvider

# import packages
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import TemplateBasedViewletManager, WeightOrderedViewletManager, viewletmanager_config
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from zope.interface import implementer, Interface
from zope.schema.fieldproperty import FieldProperty

from pyams_skin import _


@viewletmanager_config(name='pyams.widget_title', view=Interface, layer=IPyAMSLayer, provides=IWidgetTitleViewletManager)
class WidgetTitleViewletManager(WeightOrderedViewletManager):
    """Widget title extensions viewlet manager"""


@viewletmanager_config(name='pyams.toolbar', view=Interface, layer=IPyAMSLayer, provides=IToolbarViewletManager)
@template_config(template='manager.pt', layer=IPyAMSLayer)
class TableToolbarViewletManager(TemplateBasedViewletManager, WeightOrderedViewletManager):
    """Table toolbar viewlet manager"""


@template_config(template='action.pt', layer=IPyAMSLayer)
@implementer(IToolbarAction)
class ToolbarAction(Viewlet):
    """Base toolbar action class"""

    label = FieldProperty(IToolbarAction['label'])
    label_css_class = FieldProperty(IToolbarAction['label_css_class'])
    css_class = FieldProperty(IToolbarAction['css_class'])
    click_handler = FieldProperty(IToolbarAction['click_handler'])
    url = FieldProperty(IToolbarAction['url'])
    modal_target = FieldProperty(IToolbarAction['modal_target'])

    def get_url(self):
        context = getattr(self.__parent__, 'actions_context', self.context)
        return absolute_url(context, self.request, self.url)


class JsToolbarAction(ToolbarAction):
    """Javascript toolbar action"""

    def get_url(self):
        return self.url


@template_config(template='menu.pt', layer=IPyAMSLayer)
@implementer(IToolbarMenu)
class ToolbarMenu(TemplateBasedViewletManager, WeightOrderedViewletManager, Viewlet):
    """Base toolbar menu class"""

    label = FieldProperty(IToolbarMenu['label'])
    label_css_class = FieldProperty(IToolbarMenu['label_css_class'])
    css_class = FieldProperty(IToolbarMenu['css_class'])
    menu_css_class = FieldProperty(IToolbarMenu['menu_css_class'])
    click_handler = FieldProperty(IToolbarMenu['click_handler'])
    url = FieldProperty(IToolbarMenu['url'])
    modal_target = FieldProperty(IToolbarMenu['modal_target'])

    def __init__(self, context, request, view, manager=None):
        WeightOrderedViewletManager.__init__(self, context, request, view)
        Viewlet.__init__(self, context, request, view, manager)

    def update(self):
        WeightOrderedViewletManager.update(self)

    def get_url(self):
        return absolute_url(self.context, self.request, self.url)

    def render(self):
        return Viewlet.render(self)


@viewlet_config(name='pyams.addings', view=Interface, layer=IPyAMSLayer, manager=IWidgetTitleViewletManager, weight=50)
@viewletmanager_config(name='pyams.addings', view=Interface, layer=IPyAMSLayer, provides=IToolbarAddingMenu)
@implementer(IToolbarAddingMenu)
class ToolbarAddingMenu(ToolbarMenu):
    """Toolbar adding menu"""

    label = _("Add...")
    css_class = 'btn btn-xs btn-success dropdown-toggle'
    menu_css_class = 'dropdown-menu'


@viewletmanager_config(name='pyams.table-item.actions', view=ITableWithActions, layer=IPyAMSLayer,
                       provides=ITableItemColumnActionsMenu)
@implementer(ITableItemColumnActionsMenu)
class TableItemColumnActionsMenu(ToolbarMenu):
    """Table item actions menu"""

    label = _("Actions...")
    label_css_class = ''
    css_class = 'btn btn-xs btn-info dropdown-toggle'


@template_config(template='toolbar-item.pt', layer=IPyAMSLayer)
@implementer(IToolbarActionItem)
class ToolbarActionItem(Viewlet):
    """Toolbar action item class"""

    label = FieldProperty(IToolbarMenuItem['label'])
    label_css_class = FieldProperty(IToolbarMenuItem['label_css_class'])
    hint_gravity = FieldProperty(IToolbarActionItem['hint_gravity'])
    css_class = FieldProperty(IToolbarMenuItem['css_class'])
    click_handler = FieldProperty(IToolbarMenuItem['click_handler'])
    url = FieldProperty(IToolbarMenuItem['url'])
    modal_target = FieldProperty(IToolbarMenuItem['modal_target'])
    stop_propagation = FieldProperty(IToolbarMenuItem['stop_propagation'])

    def get_url(self):
        context = getattr(self.__parent__, 'actions_context', self.context)
        return absolute_url(context, self.request, self.url)


class JsToolbarActionItem(ToolbarActionItem):
    """Javascript call toolbar action item"""

    def get_url(self):
        return self.url


@template_config(template='menu-item.pt', layer=IPyAMSLayer)
@implementer(IToolbarMenuItem)
class ToolbarMenuItem(ToolbarActionItem):
    """Toolbar menu item class"""


class JsToolbarMenuItem(ToolbarMenuItem):
    """Javascript call toolbar menu item"""

    def get_url(self):
        return self.url


@template_config(template='menu-divider.pt', layer=IPyAMSLayer)
@implementer(IToolbarMenuItem)
class ToolbarMenuDivider(Viewlet):
    """Toolbar menu divider"""

    lebel = None
    css_class = 'divider'


@viewlet_config(name='pyams.actions', view=Interface, layer=IPyAMSLayer, manager=IToolbarViewletManager, weight=999)
@viewletmanager_config(name='pyams.actions', view=Interface, layer=IPyAMSLayer, provides=IContextActions)
@template_config(template='menu.pt', layer=IPyAMSLayer)
@implementer(IContextActions)
class ContextActionsViewletManager(ToolbarMenu):
    """Context actions viewlet manager"""

    label = _("Other actions...")
    label_css_class = ''
    css_class = FieldProperty(IContextActions['css_class'])


@adapter_config(name='context_actions', context=(Interface, Interface, Interface), provides=ITALESExtension)
class ContextActionsExtension(ContextRequestViewAdapter):
    """extension:context_actions(context) TALES extension"""

    def render(self, context=None):
        if context is None:
            context = self.context
        registry = self.request.registry
        provider = registry.queryMultiAdapter((context, self.request, self.view), IContentProvider,
                                              name='pyams.actions')
        if provider is not None:
            provider.update()
            return provider
