#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from pyramid.renderers import render

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.container import ITable, ITableElementEditor, ITableWithActions, ITableElementName
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces import MANAGE_PERMISSION, ICacheKeyValue
from pyams_viewlet.interfaces import IViewletManager
from z3c.table.interfaces import IColumn, IValues
from zope.container.interfaces import IContained, IContainer

# import packages
from pyams_template.template import get_view_template, template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet
from pyramid.decorator import reify
from pyramid.url import resource_url
from z3c.table.column import Column, GetAttrColumn
from z3c.table.table import Table
from zope.interface import implementer, Interface

from pyams_skin import _


@adapter_config(context=(IContained, IPyAMSLayer, ITable), provides=ITableElementEditor)
class DefaultElementEditorAdapter(ContextRequestViewAdapter):
    """Default contained element editor"""

    view_name = 'properties.html'

    @property
    def url(self):
        return resource_url(self.context, self.request, self.view_name)

    modal_target = True


def get_table_id(table, context=None):
    """Get ID for selected table"""
    if context is None:
        context = table.context
    return '{0}_{1}'.format(table.prefix, ICacheKeyValue(context))


def get_element_id(table, element, table_context=None):
    """Get ID for selected element"""
    return '{0}::{1}'.format(get_table_id(table, table_context), ICacheKeyValue(element))


def get_element_name(element):
    """Get name for selected element"""
    return getattr(element, '__name__', None)


def get_element_editor(table, element):
    """Get editor for selected element"""
    registry = table.request.registry
    return registry.queryMultiAdapter((element, table.request, table), ITableElementEditor)


@template_config(template='templates/table.pt', layer=IPyAMSLayer)
@implementer(ITable)
class BaseTable(Table):
    """Custom table"""

    def __init__(self, context, request):
        super(BaseTable, self).__init__(self.get_context(context), request)

    @reify
    def id(self):
        return get_table_id(self, self.context)

    def get_element_id(self, element):
        return '{0}::{1}'.format(self.id, ICacheKeyValue(element))

    title = _("Container elements")

    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight datatable'}

    @classmethod
    def get_context(cls, context):
        return context

    @property
    def data_attributes(self):
        return {
            'table': {
                'id': self.id,
            },
            'tr': {
                'id': lambda x, col: self.get_element_id(x),
                'data-ams-element-name': lambda x, col: get_element_name(x),
                'data-ams-url': lambda x, col: getattr(get_element_editor(self, x), 'url', None),
                'data-toggle': lambda x, col: 'modal' if getattr(get_element_editor(self, x), 'modal_target',
                                                                 None) else None
            },
            'th': {
                'data-ams-column-name': lambda x, col: x.__name__,
                'data-ams-datatable-sortable': self.get_sortable_column,
                'data-ams-datatable-stype': self.get_column_type
            }
        }

    batchSize = 10000
    startBatchingAt = 10000

    def getBatchSize(self):
        return int(self.request.params.get(self.prefix + '-batchSize', self.batchSize))

    def getBatchStart(self):
        return int(self.request.params.get(self.prefix + '-batchStart', self.batchStart))

    def getSortOn(self):
        return self.request.params.get(self.prefix + '-sortOn', self.sortOn)

    def getSortOrder(self):
        return self.request.params.get(self.prefix + '-sortOrder', self.sortOrder)

    @staticmethod
    def get_sortable_column(column, _=None):
        return getattr(column, 'dt_sortable', None)

    @staticmethod
    def get_column_type(column, _=None):
        return getattr(column, 'dt_sort_type', None)

    @staticmethod
    def check_data_attribute(attribute, source, column=None):
        if callable(attribute):
            return attribute(source, column)
        else:
            return str(attribute)

    def get_data_attributes(self, element, source, column=None):
        attrs = self.data_attributes.get(element)
        if attrs:
            result = ''
            for key, value in attrs.items():
                checked_value = self.check_data_attribute(value, source, column)
                if checked_value is not None:
                    result += ' {0}="{1}"'.format(key, checked_value)
            return result
        else:
            return ''

    render = get_view_template()

    def getSelectedRowClass(self, row, cssClass=None):
        klass = self.cssClasses.get('tr.selected')
        if callable(klass):
            klass = klass(*row)
        if klass and cssClass:
            klass = '{0} {1}'.format(klass, cssClass)
        elif cssClass:
            klass = cssClass
        return klass or ''

    def renderTable(self):
        return super(BaseTable, self).renderTable() \
            .replace('<table', '<table %s' % self.get_data_attributes('table', self))

    def renderEmptyTable(self, message=''):
        return render('templates/table-empty.pt',
                      {'view': self,
                       'context': self.context,
                       'message': message},
                      request=self.request)

    def renderRow(self, row, cssClass=None):
        cssClass = self.getSelectedRowClass(row[0], cssClass)
        return super(BaseTable, self).renderRow(row, cssClass) \
            .replace('<tr', '<tr %s' % self.get_data_attributes('tr', row[0][0]))

    def renderHeadCell(self, column):
        return super(BaseTable, self).renderHeadCell(column) \
            .replace('<th', '<th %s' % self.get_data_attributes('th', column))

    def renderCell(self, item, column, colspan=0):
        return super(BaseTable, self).renderCell(item, column, colspan) \
            .replace('<td', '<td %s' % self.get_data_attributes('td', item, column))


@adapter_config(context=(IContainer, IPyAMSLayer, BaseTable), provides=IValues)
class ContainerValuesAdapter(ContextRequestViewAdapter):
    """Default container values adapter"""

    @property
    def values(self):
        return list(self.context.values())


class I18nColumn(object):
    """Column with I18n header"""

    _header = None
    request = None

    @property
    def header(self):
        if not self._header:
            return ''
        return self.request.localizer.translate(self._header)


class I18nValueColumn(GetAttrColumn):
    """Column with I18n value"""

    def getValue(self, obj):
        if obj is not None and self.attrName is not None:
            return II18n(obj).query_attribute(self.attrName, request=self.request)
        return self.defaultValue


def get_object_name(context, request, view=None):
    """Get required object name"""
    registry = request.registry
    adapter = None
    if view is not None:
        adapter = registry.queryMultiAdapter((context, request, view), ITableElementName)
    if adapter is None:
        adapter = registry.queryMultiAdapter((context, request), ITableElementName)
    if adapter is None:
        adapter = registry.queryAdapter(context, ITableElementName)
    if adapter is not None:
        return adapter.name


@adapter_config(name='name', context=(Interface, IPyAMSLayer, BaseTable), provides=IColumn)
class NameColumn(I18nColumn, GetAttrColumn):
    """Container name column"""

    _header = _("Name")
    attrName = '__name__'
    weight = 10

    dt_sort_type = 'string'

    def getValue(self, obj):
        name = get_object_name(obj, self.request, self.table)
        if name is None:
            name = super(NameColumn, self).getValue(obj)
        return name


class ActionColumn(Column):
    """Base action icon column"""

    header = ''
    icon_class = 'fa fa-fw fa-edit'
    icon_hint = _("Properties")
    cssClasses = {'th': 'action',
                  'td': 'action'}
    url = "#"
    target = '#content'
    modal_target = False
    checker = None
    permission = None

    dt_sortable = 'false'

    def renderCell(self, item):
        if not self.has_permission(item):
            return ''
        if self.checker:
            if callable(self.checker):
                checked = self.checker(item)
            else:
                checked = self.checker
            if not checked:
                return ''
        return '''<a class="hint" title="{title}" href="{url}"
                     data-ams-stop-propagation="true"
                     {target} {modal} data-ams-hint-gravity="e">
            {icon}
        </a>'''.format(title=self.get_icon_hint(item),
                       url=self.get_url(item),
                       target='data-ams-target="{0}"'.format(self.target) if self.target else '',
                       modal='data-toggle="modal"' if self.modal_target else '',
                       icon=self.get_icon(item))

    def has_permission(self, item):
        if not self.permission:
            return True
        return self.request.has_permission(self.permission, item)

    def get_url(self, item):
        return absolute_url(item, self.request, self.url)

    def get_icon(self, item):
        return '<i class="{icon_class}"></i>'.format(icon_class=self.get_icon_class(item))

    def get_icon_class(self, item):
        return self.icon_class

    def get_icon_hint(self, item):
        return self.request.localizer.translate(self.icon_hint)


class JsActionColumn(ActionColumn):
    """Javascript function caller action column"""

    def get_url(self, item):
        return self.url


class ImageColumn(ActionColumn):
    """Simple image column"""

    def renderCell(self, item):
        klass = self.get_icon_class(item)
        hint = self.get_icon_hint(item)
        if klass.startswith('fa-'):
            img = '<i class="fa fa-fw {icon_class} {hint_class}" {title}></i>'.format(
                icon_class=klass,
                hint_class='hint opaque align-base' if hint else '',
                title='title="{hint}"'.format(hint=hint) if hint else '')
        elif klass.startswith('/'):
            img = '<img src="{icon_class}" class="{img_class}" {title} />'.format(
                icon_class=klass,
                img_class='hint opaque align-base' if hint else '',
                title='title="{hint}"'.format(hint=hint) if hint else '')
        else:
            img = klass
        return img


class SorterColumn(ActionColumn):
    """Rows sorter column"""

    cssClasses = {'th': 'action',
                  'td': 'action sorter'}

    icon_class = 'fa fa-fw fa-sort'
    icon_hint = _("Click and drag to sort rows")

    url = '#'
    weight = 1

    def get_url(self, item):
        return '#'


class AttributeSwitcherColumn(JsActionColumn):
    """Switcher column"""

    cssClasses = {'th': 'action',
                  'td': 'action switcher'}

    icon_hint = _("Switch element attribute")

    switch_attribute = None

    on_icon_class = 'fa fa-fw fa-eye'
    off_icon_class = 'fa fa-fw fa-eye-slash text-danger'

    url = 'MyAMS.container.switchElementAttribute'
    weight = 5

    def get_icon(self, item):
        if getattr(item, self.switch_attribute):
            icon_class = self.on_icon_class
        else:
            icon_class = self.off_icon_class
        return '<i class="{0}"></i>'.format(icon_class)

    def renderCell(self, item):
        permission = self.permission
        if permission and not self.request.has_permission(permission, context=item):
            return self.get_icon(item)
        else:
            return super(AttributeSwitcherColumn, self).renderCell(item)


class VisibilitySwitcherColumn(AttributeSwitcherColumn):
    """Visibility switcher column"""

    switch_attribute = 'visible'

    icon_hint = _("Switch element visibility")

    url = 'MyAMS.container.switchElementVisibility'


@adapter_config(name='actions', context=(Interface, IPyAMSLayer, ITableWithActions), provides=IColumn)
class MenuActionsColumn(I18nColumn, Column):
    """Menu actions column"""

    _header = _("Actions")
    cssClasses = {'th': 'actions-menu',
                  'td': 'nowrap'}
    weight = 9000

    def renderCell(self, item):
        registry = self.request.registry
        viewlet = registry.queryMultiAdapter((item, self.request, self.table), IViewletManager,
                                             name='pyams.table-item.actions')
        if viewlet is not None:
            viewlet.update()
            return viewlet.render()
        else:
            return ''


class TrashColumn(JsActionColumn):
    """Trash action icon column"""

    url = 'MyAMS.container.deleteElement'
    icon_class = 'fa fa-fw fa-trash'
    icon_hint = _("Delete object")
    permission = MANAGE_PERMISSION
    weight = 9999


@template_config(template='templates/table-viewlet.pt', layer=IPyAMSLayer)
class TableViewlet(Viewlet):
    """Table viewlet"""

    table_class = None

    def __init__(self, context, request, view, manager):
        super(TableViewlet, self).__init__(context, request, view, manager)
        self.table = self.table_class(context, request)

    def update(self):
        super(TableViewlet, self).update()
        self.table.update()
