/**
 * MyAMS standard plug-ins loader
 *
 * Only basic JQuery, Bootstrap and MyAMS javascript extensions are typically loaded from main page.
 * Other JQuery plug-ins may be loaded dynamically.
 * Several JQuery extension plug-ins are already included and pre-configured by MyAMS. Other external
 * plug-ins can be defined and loaded dynamically using simple "data" attributes.
 *
 * WARNING: any plug-in implicated into a form submit process (like JQuery-form or JQuery-progressbar)
 * must be loaded in a synchronous way. Otherwise, if you use named buttons to submit your forms,
 * dynamic hidden input fields created by JQuery-validate plug-in will be removed from the form
 * before the form is submitted!
 */
(function($, globals) {

	var MyAMS = globals.MyAMS,
		ams = MyAMS;

	MyAMS.plugins = {

		/**
		 * Container of enabled plug-ins
		 */
		enabled: {},

		/**
		 * Initialize list of content plug-ins
		 */
		init: function(element) {

			// Initialize custom data attributes
			ams.plugins.initData(element);

			// Check for disabled plug-ins
			var disabled = [];
			$('[data-ams-plugins-disabled]', element).each(function() {
				var plugins = $(this).data('ams-plugins-disabled').split(/\s+/);
				for (var index=0; index < plugins.length; index++) {
					disabled.push(plugins[index]);
				}
			});

			// Scan new element for plug-ins
			var plugins = {};
			var name;

			// Inner plug-in register function
			function _registerPlugin(name, new_plugin) {
				if (plugins.hasOwnProperty(name)) {
					var plugin = plugins[name];
					plugin.css = plugin.css || new_plugin.css;
					plugin.callbacks.push({
						callback: new_plugin.callback,
						context: new_plugin.context
					});
					if (new_plugin.register) {
						plugin.register = true;
					}
					if (new_plugin.async === false) {
						plugin.async = false;
					}
				} else {
					plugins[name] = {
						src: new_plugin.src,
						css: new_plugin.css,
						callbacks: [{
							callback: new_plugin.callback,
							context: new_plugin.context
						}],
						register: new_plugin.register,
						async: new_plugin.async
					};
				}
				if (new_plugin.css) {
					ams.getCSS(new_plugin.css, name + '_css');
				}
			}

			$('[data-ams-plugins]', element).each(function() {

				var source = $(this);
				var amsPlugins = source.data('ams-plugins');
				if (typeof(amsPlugins) === 'string') {
					var names = source.data('ams-plugins').split(/\s+/);
					for (var index = 0; index < names.length; index++) {
						name = names[index];
						var newPlugin = {
							src: source.data('ams-plugin-' + name + '-src'),
							css: source.data('ams-plugin-' + name + '-css'),
							callback: source.data('ams-plugin-' + name + '-callback'),
							context: source,
							register: source.data('ams-plugin-' + name + '-register'),
							async: source.data('ams-plugin-' + name + '-async')
						};
						_registerPlugin(name, newPlugin);
					}
				} else {
					for (name in amsPlugins) {
						if (!amsPlugins.hasOwnProperty(name)) {
							continue;
						}
						_registerPlugin(name, amsPlugins[name]);
					}
				}
			});

			// Inner plug-in loader function
			var plugin;

			function _loadPlugin(reload) {
				var index;
				var callbacks = plugin.callbacks,
					callback;
				if (callbacks && callbacks.length) {
					for (index=0; index < callbacks.length; index++) {
						callback = callbacks[index];
						callback.callback = ams.getFunctionByName(callback.callback);
						if (plugin.register !== false) {
							var enabled = ams.plugins.enabled;
							if (enabled.hasOwnProperty(name)) {
								enabled[name].push(callback);
							} else {
								enabled[name] = [callback];
							}
						}
					}
				} else {
					if (plugin.register !== false) {
						ams.plugins.enabled[name] = null;
					}
				}
				// If running in async mode, newly registered plug-ins are run
				// before callback is called so we call plug-in manually
				if ((reload !== true) && callbacks && callbacks.length && (plugin.async !== false)) {
					for (index=0; index < callbacks.length; index++) {
						callback = callbacks[index];
						ams.executeFunctionByName(callback.callback, element, callback.context);
					}
				}
			}

			function _checkPluginContext() {
				// Update context for an already loaded plug-in
				var enabled = ams.plugins.enabled[name];
				// Clean all plug-in contexts
				for (index=0; index < enabled.length; index++) {
					var callback = enabled[index];
					if (callback && callback.context && !ams.isInDOM(callback.context)) {
						enabled[index] = null;
					}
				}
			}

			for (name in plugins) {
				if (!plugins.hasOwnProperty(name)) {
					continue;
				}
				plugin = plugins[name];
				if (ams.plugins.enabled[name] === undefined) {
					ams.getScript(plugin.src, _loadPlugin, {
						async: plugin.async === undefined ? true : plugin.async
					});
				} else {
					_checkPluginContext();
					_loadPlugin(true);
				}
			}

			// Run all enabled plug-ins
			for (var index in ams.plugins.enabled) {
				if (!ams.plugins.enabled.hasOwnProperty(index)) {
					continue;
				}
				if (disabled.indexOf(index) >= 0) {
					continue;
				}
				var callbacks = ams.plugins.enabled[index];
				if (callbacks) {
					switch (typeof(callbacks)) {
						case 'function':
							callbacks(element);
							break;
						default:
							for (var cbIndex = 0; cbIndex < callbacks.length; cbIndex++) {
								var callback = callbacks[cbIndex];
								switch (typeof(callback)) {
									case 'function':
										callback(element);
										break;
									default:
										if (callback && callback.callback) {
											callback.callback(callback.context);
										}
								}
							}
					}
				}
			}
		},

		/**
		 * Data initializer
		 * This plug-in converts a single JSON "data-ams-data" attribute into a set of several equivalent "data-" attributes.
		 * This way of defining data attributes can be used with HTML templates engines which don't allow you
		 * to create dynamic attributes easily.
		 */
		initData: function(element) {
			$('[data-ams-data]', element).each(function() {
				var dataElement = $(this);
				var data = dataElement.data('ams-data');
				if (data) {
					for (var name in data) {
						if (data.hasOwnProperty(name)) {
							var elementData = data[name];
							if (typeof(elementData) !== 'string') {
								elementData = JSON.stringify(elementData);
							}
							dataElement.attr('data-' + name, elementData);
						}
					}
				}
			});
		},

		/**
		 * Register a new plug-in through Javascript instead of HTML data attributes
		 *
		 * @plugin: plugin function caller or object containing plug-in properties
		 * @name: if @plugin is a function, defines plug-in name
		 * @callback: a callback function which can be called after plug-in registry
		 */
		register: function(plugin, name, callback) {
			if (typeof(name) === 'function') {
				callback = name;
				name = null;
			}
			name = name || plugin.name;
			if (ams.plugins.enabled.indexOf(name) >= 0) {
				if (console) {
					console.warn && console.warn("Plugin " + name + " is already registered!");
				}
				return;
			}
			if (typeof(plugin) === 'object') {
				var src = plugin.src;
				if (src) {
					ams.ajax.check(plugin.callback, src, function(first_load) {
						if (first_load) {
							ams.plugins.enabled[name] = ams.getFunctionByName(plugin.callback);
							if (plugin.css) {
								ams.getCSS(plugin.css, name + '_css');
							}
							if (callback) {
								ams.executeFunctionByName(callback);
							}
						}
					});
				} else {
					ams.plugins.enabled[name] = ams.getFunctionByName(plugin.callback);
					if (plugin.css) {
						ams.getCSS(plugin.css, name + '_css');
					}
					if (callback) {
						ams.executeFunctionByName(callback);
					}
				}
			} else if (typeof(plugin) === 'function') {
				ams.plugins.enabled[name] = plugin;
				if (callback) {
					ams.executeFunctionByName(callback);
				}
			}
		}
	};

})(jQuery, this);
