/**
 * MyAMS page initialization
 * This code is called once to register global events and callbacks
 */
(function($, globals) {

	var MyAMS = globals.MyAMS,
		ams = MyAMS;

	MyAMS.initPage = function() {

		var body = $('body');

		/* Init main components */
		ams.root = body;
		ams.leftPanel = $('#left-panel');
		ams.shortcuts = $('#shortcuts');
		ams.plugins.initData(body);

		// Init main AJAX events
		var xhr = $.ajaxSettings.xhr;
		$.ajaxSetup({
			beforeSend: function(request, options) {
				// Check CSRF token for unsafe methods
				if (ams.safeMethods.indexOf(options.type) < 0) {
					if (globals.Cookies !== undefined) {
						var token = Cookies.get(ams.csrfCookieName);
						if (token) {
							request.setRequestHeader(ams.csrfHeaderName, token);
						}
					}
				}
			},
			progress: ams.ajax.progress,
			progressUpload: ams.ajax.progress,
			xhr: function() {
				var request = xhr();
				if (request && (typeof(request.addEventListener) === "function")) {
					var that = this;
					if (that && that.progress) {
						request.addEventListener("progress", function (evt) {
							that.progress(evt);
						}, false);
					}
				}
				return request;
			}
		});
		$(document).ajaxStart(ams.ajax.start);
		$(document).ajaxStop(ams.ajax.stop);
		$(document).ajaxError(ams.error.ajax);

		// Check for minified state in local storage
		var state = globals.localStorage && globals.localStorage.getItem('window-state');
		if (state) {
			body.addClass(state);
		}

		// Check mobile/desktop
		if (!ams.isMobile) {
			ams.root.addClass('desktop-detected');
			ams.device = 'desktop';
		} else {
			ams.root.addClass('mobile-detected');
			ams.device = 'mobile';
			if (ams.enableFastclick) {
				ams.ajax.check($.fn.noClickDelay,
							   ams.baseURL + '/ext/jquery-smartclick' + ams.devext + '.js',
							   function() {
								   $('NAV UL A').noClickDelay();
								   $('A', '#hide-menu').noClickDelay();
							   });
			}
		}

		// Switch shortcuts
		$('#show-shortcuts').click(function(e) {
			if (ams.shortcuts.is(":visible")) {
				ams.skin._hideShortcutButtons();
			} else {
				ams.skin._showShortcutButtons();
			}
			e.preventDefault();
		});
		ams.shortcuts.click(function(e) {
			ams.skin._hideShortcutButtons();
		});

		$(document).mouseup(function(e) {
			if (!ams.shortcuts.is(e.target) &&
				ams.shortcuts.has(e.target).length === 0) {
				ams.skin._hideShortcutButtons();
			}
		});

		// Show & hide mobile search field
		$('#search-mobile').click(function() {
			ams.root.addClass('search-mobile');
		});

		$('#cancel-search-js').click(function() {
			ams.root.removeClass('search-mobile');
		});

		// Activity badge
		$('.activity-button', '#user-activity').click(function(e) {
			var activity = $('#user-activity');
			var dropdown = $('.ajax-dropdown', activity);
			if (!dropdown.is(':visible')) {
				dropdown.css('left', - dropdown.innerWidth() + activity.innerWidth())
						.fadeIn(150);
				activity.addClass('active');
			} else {
				dropdown.fadeOut(150);
				activity.removeClass('active');
			}
			e.preventDefault();
		});
		ams.skin.checkNotification();

		$(document).mouseup(function(e) {
			var dropdown = $('.ajax-dropdown');
			if (!dropdown.is(e.target) &&
				dropdown.has(e.target).length === 0) {
				dropdown.fadeOut(150)
						.prev().removeClass("active");
			}
		});

		$('input[name="activity"]').change(function(e) {
			var href = $(this).data('ams-url');
			if (href) {
				e.preventDefault();
				e.stopPropagation();
				var hrefGetter = ams.getFunctionByName(href);
				if (typeof(hrefGetter) === 'function') {
					href = hrefGetter.call(this);
				}
				if (typeof(href) === 'function') {
					// Javascript function call
					href.call(this);
				} else {
					var container = $('.ajax-notifications');
					ams.skin.loadURL(href, container);
				}
			}
		});

		// Logout button
		$('a', '#user-menu LI.logout').click(function(e) {
			e.preventDefault();
			e.stopPropagation();
			//get the link
			ams.loginURL = $(this).attr('href');
			// ask verification
			ams.skin.bigBox({
				title : "<i class='fa fa-sign-out txt-color-orangeDark'></i> " + ams.i18n.LOGOUT +
						" <span class='txt-color-orangeDark'><strong>" + $('#show-shortcut').text() + "</strong></span> ?",
				content : ams.i18n.LOGOUT_COMMENT,
				buttons : ams.i18n.BTN_YES_NO
			}, function(ButtonPressed) {
				if (ButtonPressed === ams.i18n.BTN_YES) {
					ams.root.addClass('animated fadeOutUp');
					setTimeout(ams.skin.logout, 1000);
				}
			});
		});

		// Initialize left nav
		var nav = $('nav');
		$('UL', nav).myams_menu({
			accordion : nav.data('ams-menu-accordion') !== false,
			speed : ams.menuSpeed
		});

		// Left navigation hide button
		$('#hide-menu').find('>:first-child >A').click(function(e) {
			body.toggleClass("hidden-menu");
			if (globals.localStorage) {
				if (body.hasClass('hidden-menu')) {
					globals.localStorage.setItem('window-state', 'hidden-menu');
				} else {
					globals.localStorage.setItem('window-state', '');
				}
			}
			e.preventDefault();
		});

		// Left navigation collapser
		$('.minifyme').click(function(e) {
			body.toggleClass("minified");
			if (globals.localStorage) {
				if (body.hasClass('minified')) {
					globals.localStorage.setItem('window-state', 'minified');
				} else {
					globals.localStorage.setItem('window-state', '');
				}
			}
			$(this).effect("highlight", {}, 500);
			e.preventDefault();
		});

		// Reset widgets
		$('#refresh').click(function(e) {
			ams.skin.bigBox({
				title: "<i class='fa fa-refresh' style='color: green'></i> " + ams.i18n.CLEAR_STORAGE_TITLE,
				content: ams.i18n.CLEAR_STORAGE_CONTENT,
				buttons: '['+ams.i18n.BTN_CANCEL+']['+ams.i18n.BTN_OK+']'
			}, function(buttonPressed) {
				if (buttonPressed === ams.i18n.BTN_OK && localStorage) {
					localStorage.clear();
					location.reload();
				}
			});
			e.preventDefault();
		});

		// Check active pop-overs
		body.on('click', function(e) {
			var element = $(this);
			if (!element.is(e.target) &&
				element.has(e.target).length === 0 &&
				$('.popover').has(e.target).length === 0) {
				element.popover('hide');
			}
		});

		// Resize events
		ams.ajax.check($.resize,
					   ams.baseURL + 'ext/jquery-resize' + ams.devext + '.js',
					   function() {
						   $('#main').resize(function() {
							   ams.skin._setPageHeight();
							   ams.skin._checkMobileWidth();
						   });
						   nav.resize(function() {
							   ams.skin._setPageHeight();
						   });
					   });

		// Init AJAX navigation
		if (ams.ajaxNav) {
			$(document).on('click', 'a[href="#"]', function(e) {
				e.preventDefault();
			});
			$(document).on('click', 'a[href!="#"]:not([data-toggle]), [data-ams-url]:not([data-toggle])', function(e) {
				var link = $(e.currentTarget);
				var handlers = link.data('ams-disabled-handlers');
				if ((handlers === true) || (handlers === 'click') || (handlers === 'all')) {
					return;
				}
				var href = link.attr('href') || link.data('ams-url');
				if (!href || href.startsWith('javascript') || link.attr('target') || (link.data('ams-context-menu') === true)) {
					return;
				}
				e.preventDefault();
				e.stopPropagation();

				var url,
					target,
					params;
				if (href.indexOf('?') >= 0) {
					url = href.split('?');
					target = url[0];
					params = url[1].unserialize();
				} else {
					target = href;
					params = undefined;
				}
				var hrefGetter = ams.getFunctionByName(target);
				if (typeof(hrefGetter) === 'function') {
					href = hrefGetter.call(link, params);
				}
				if (typeof(href) === 'function') {
					// Javascript function call
					href.call(link, params);
				} else {
					// Standard AJAX or browser URL call
					// Convert %23 chars to #
					href = href.replace(/\%23/, '#');
					if (e.ctrlKey) {
						window.open(href);
					} else {
						var target = link.data('ams-target');
						if (target) {
							ams.form.confirmChangedForm(target, function () {
								ams.skin.loadURL(href, target, link.data('ams-link-options'), link.data('ams-link-callback'));
							});
						} else {
							ams.form.confirmChangedForm(function () {
								if (href.startsWith('#')) {
									if (href !== location.hash) {
										if (ams.root.hasClass('mobile-view-activated')) {
											ams.root.removeClass('hidden-menu');
											window.setTimeout(function () {
												window.location.hash = href;
											}, 50);
										} else {
											window.location.hash = href;
										}
									}
								} else {
									window.location = href;
								}
							});
						}
					}
				}
			});
			$(document).on('click', 'a[target="_blank"]', function(e) {
				e.preventDefault();
				var target = $(e.currentTarget);
				window.open(target.attr('href'));
				ams.stats.logEvent(target.data('ams-stats-category') || 'Navigation',
								   target.data('ams-stats-action') || 'External',
								   target.data('ams-stats-label') || target.attr('href'));
			});
			$(document).on('click', 'a[target="_top"]', function(e) {
				e.preventDefault();
				ams.form.confirmChangedForm(function() {
					window.location = $(e.currentTarget).attr('href');
				});
			});

			// Check URL when hash changed
			$(window).on('hashchange', ams.skin.checkURL);
		}

		// Initialize modal dialogs links
		$(document).off('click.modal')
				   .on('click', '[data-toggle="modal"]', function(e) {
			var source = $(this);
			var handlers = source.data('ams-disabled-handlers');
			if ((handlers === true) || (handlers === 'click') || (handlers === 'all')) {
				return;
			}
			if (source.data('ams-context-menu') === true) {
				return;
			}
			if (source.data('ams-stop-propagation') === true) {
				e.stopPropagation();
			}
			e.preventDefault();
			ams.dialog.open(source);
			if (source.parents('#shortcut').exists()) {
				setTimeout(ams.skin._hideShortcutButtons, 300);
			}
		});

		// Initialize form buttons
		$(document).on('click', 'button[type="submit"], button.submit', function() {
			var button = $(this);
			$(button.get(0).form).data('ams-submit-button', button);
		});

		// Cancel clicks on readonly checkbox
		$(document).on('click', 'input[type="checkbox"][readonly]', function() {
			return false;
		});

		// Initialize custom click handlers
		$(document).on('click', '[data-ams-click-handler]', function(event) {
			var source = $(this);
			var handlers = source.data('ams-disabled-handlers');
			if ((handlers === true) || (handlers === 'click') || (handlers === 'all')) {
				return;
			}
			var data = source.data();
			if (data.amsClickHandler) {
				if ((data.amsStopPropagation === true) || (data.amsClickStopPropagation === true)) {
					event.stopPropagation();
				}
				if (data.amsClickKeepDefault !== true) {
					event.preventDefault();
				}
				var clickHandlers = data.amsClickHandler.split(/\s+/);
				for (var index=0; index < clickHandlers.length; index++) {
					var callback = ams.getFunctionByName(clickHandlers[index]);
					if (callback !== undefined) {
						callback.call(source, event, data.amsClickHandlerOptions);
					}
				}
			}
		});

		// Initialize custom change handlers
		$(document).on('change', '[data-ams-change-handler]', function(event) {
			var source = $(this);
			// Disable change handlers for readonly inputs
			// These change handlers are activated by IE!!!
			if (source.prop('readonly')) {
				return;
			}
			var handlers = source.data('ams-disabled-handlers');
			if ((handlers === true) || (handlers === 'change') || (handlers === 'all')) {
				return;
			}
			var data = source.data();
			if (data.amsChangeHandler) {
				if ((data.amsStopPropagation === true) || (data.amsChangeStopPropagation === true)) {
					event.stopPropagation();
				}
				if (data.amsChangeKeepDefault !== true) {
					event.preventDefault();
				}
				var changeHandlers = data.amsChangeHandler.split(/\s+/);
				for (var index=0; index < changeHandlers.length; index++) {
					var callback = ams.getFunctionByName(changeHandlers[index]);
					if (callback !== undefined) {
						callback.call(source, event, data.amsChangeHandlerOptions);
					}
				}
			}
		});

		// Submit form when CTRL+Enter key is pressed in textarea
		$(document).on('keydown', 'textarea', function(e) {
			if ((e.keyCode === 10 || e.keyCode === 13) && (e.ctrlKey || e.metaKey)) {
				$(this).closest('form').submit();
			}
		});

		// Notify reset to update Select2 widgets
		$(document).on('reset', 'form', function(e) {
			var form = $(this);
			setTimeout(function() {
				$('.alert-danger, SPAN.state-error', form).not('.persistent').remove();
				$('LABEL.state-error', form).removeClass('state-error');
				$('INPUT.select2[type="hidden"]', form).each(function() {
					var input = $(this);
					var select = input.data('select2');
					var value = input.data('ams-select2-input-value');
					if (value) {
						input.select2('val', value.split(select.opts.separator));
					}
				});
				form.find('.select2').trigger('change');
				$('[data-ams-reset-callback]', form).each(function() {
					var element = $(this);
					var data = element.data();
					var callback = ams.getFunctionByName(data.amsResetCallback);
					if (callback !== undefined) {
						callback.call(form, element, data.amsResetCallbackOptions);
					}
				});
			}, 10);
			ams.form.setFocus(form);
		});

		// Initialize custom reset handlers
		$(document).on('reset', '[data-ams-reset-handler]', function(e) {
			var form = $(this);
			var data = form.data();
			if (data.amsResetHandler) {
				if (data.amsResetKeepDefault !== true) {
					e.preventDefault();
				}
				var callback = ams.getFunctionByName(data.amsResetHandler);
				if (callback !== undefined) {
					callback.call(form, data.amsResetHandlerOptions);
				}
			}
		});

		// Initialize custom event on click
		$(document).on('click', '[data-ams-click-event]', function(e) {
			var source = $(this);
			$(e.target).trigger(source.data('ams-click-event'), source.data('ams-click-event-options'));
		});

		// Handle update on file upload placeholder
		$(document).on('change', 'input[type="file"]', function(e) {
			e.preventDefault();
			var input = $(this);
			var button = input.parent('.button');
			if (button.exists() && button.parent().hasClass('input-file')) {
				button.next('input[type="text"]').val(input.val());
			}
		});

		// Always blur readonly inputs
		$(document).on('focus', 'input[readonly="readonly"]', function() {
			$(this).blur();
		});

		// Prevent bootstrap dialog from blocking TinyMCE focus
		$(document).on('focusin', function(e) {
			if ($(e.target).closest('.mce-window').length) {
				e.stopImmediatePropagation();
			}
		});

		// Disable clicks on disabled tabs
		$(document).on("click", '.nav-tabs a[data-toggle=tab]', function(e) {
			if ($(this).parent('li').hasClass("disabled")) {
				e.preventDefault();
				return false;
			}
		});

		// Automatically set orientation of dropdown menus
		$(document).on('show.bs.dropdown', '.btn-group', function() {
			var menu = $(this);
			var ul = menu.children('.dropdown-menu');
			var menuRect = menu.get(0).getBoundingClientRect();
			var position = menuRect.top;
			var buttonHeight = menuRect.height;
			var menuHeight = ul.outerHeight();
			if (position > menuHeight && $(window).height() - position < buttonHeight + menuHeight) {
				menu.addClass("dropup");
			}
		}).on('hidden.bs.dropdown', '.btn-group', function() {
			// always reset after close
			$(this).removeClass('dropup');
		});

		// Enable tabs dynamic loading
		$(document).on('show.bs.tab', function(e) {
			var link = $(e.target);
			if (link.exists() && (link.get(0).tagName !== 'A')) {
				link = $('a[href]', link);
			}
			var data = link.data();
			if (data && data.amsUrl) {
				if (data.amsTabLoaded) {
					return;
				}
				link.append('<i class="fa fa-spin fa-cog margin-left-5"></i>');
				ams.skin.loadURL(data.amsUrl, link.attr('href'), {
					afterLoadCallback: function() {
						if (data.amsTabLoadOnce) {
							link.data('ams-tab-loaded', true);
						}
						$('i', link).remove();
					},
					afterErrorCallback: function() {
						$('i', link).remove();
					}
				});
			}
		});

		// Check modal form dialogs on close
		$(document).on('hide.bs.modal', function(e) {
			var modal = $(e.target);
			ams.form.confirmChangedForm(modal, function() {
				// Confirm closing if OK
				var bsModal = modal.data('modal') || modal.data('bs.modal');
				if (bsModal) {
					bsModal.isShown = true;
				}
				return true;
			}, function() {
				// Prevent closing if cancelled
				e.preventDefault();
				return false;
			});
		});

		// Enable custom MyAMS refresh events
		$(document).on('myams.refresh', function(event, options) {
			MyAMS.executeFunctionByName(options.handler || MyAMS.skin.refreshContent, event.target, options);
		});

		// Init page content
		ams.initContent(document);
		if (ams.ajaxNav && nav.exists()) {
			ams.skin.checkURL();
		}
		ams.form.setFocus(document);

		// Add unload event listener to check for modified forms
		$(window).on('beforeunload', ams.form.checkBeforeUnload);

	};


	/**
	 * Main content plug-ins initializer
	 * This code is called to initialize plugins, callbacks and events listeners each time an HTML content
	 * is loaded dynamically from remote server.
	 */
	MyAMS.initContent = function(element) {

		// Remove left tips
		$('.tipsy').remove();

		// Activate tooltips and popovers
		$("[rel=tooltip]", element).tooltip();
		$("[rel=popover]", element).popover();

		// Activate popovers with hover states
		$("[rel=popover-hover]", element).popover({
			trigger : "hover"
		});

		// Init registered plug-ins and callbacks
		ams.plugins.init(element);
		ams.callbacks.init(element);
		ams.events.init(element);
		ams.form.init(element);

		// Initialize widgets
		if (ams.device === 'desktop') {
			ams.skin._initDesktopWidgets(element);
		} else {
			ams.skin._initMobileWidgets(element);
		}
		ams.skin._setPageHeight();

	};


	/**
	 * MyAMS locale strings
	 */
	MyAMS.i18n = {

		INFO: "Information",
		WARNING: "!! WARNING !!",
		ERROR: "ERROR: ",

		LOADING: "Loading...",
		PROGRESS: "Processing",

		WAIT: "Please wait!",
		FORM_SUBMITTED: "This form was already submitted...",
		NO_SERVER_RESPONSE: "No response from server!",

		ERROR_OCCURED: "An error occured!",
		ERRORS_OCCURED: "Some errors occured!",

		BAD_LOGIN_TITLE: "Bad login!",
		BAD_LOGIN_MESSAGE: "Your anthentication credentials didn't allow you to open a session; " +
						   "please check your credentials or contact administrator.",

		CONFIRM: "Confirm",
		CONFIRM_REMOVE: "Removing this content can't be undone. Do you confirm?",

		CLEAR_STORAGE_TITLE: "Clear Local Storage",
		CLEAR_STORAGE_CONTENT: "Would you like to RESET all your saved widgets and clear LocalStorage?",

		BTN_OK: "OK",
		BTN_CANCEL: "Cancel",
		BTN_OK_CANCEL: "[OK][Cancel]",
		BTN_YES: "Yes",
		BTN_NO: "No",
		BTN_YES_NO: "[Yes][No]",

		CLIPBOARD_COPY: "Copy to clipboard with Ctrl+C, and Enter",
		CLIPBOARD_CHARACTER_COPY_OK: "Character copied to clipboard",
		CLIPBOARD_TEXT_COPY_OK: "Text copied to clipboard",

		FORM_CHANGED_WARNING: "Some changes were not saved. These updates will be lost if you leave this page.",
		DELETE_WARNING: "This change can't be undone. Are you sure that you want to delete this element?",
		NO_UPDATE: "No changes were applied.",
		DATA_UPDATED: "Data successfully updated.",

		HOME: "Home",
		LOGOUT: "Logout?",
		LOGOUT_COMMENT: "You can improve your security further after logging out by closing this opened browser",

		SELECT2_PLURAL: 's',
		SELECT2_MATCH: "One result is available, press enter to select it.",
		SELECT2_MATCHES: " results are available, use up and down arrow keys to navigate.",
		SELECT2_NOMATCHES: "No matches found",
		SELECT2_SEARCHING: "Searching...",
		SELECT2_LOADMORE: "Loading more results...",
		SELECT2_INPUT_TOOSHORT: "Please enter {0} more character{1}",
		SELECT2_INPUT_TOOLONG: "Please delete {0} character{1}",
		SELECT2_SELECTION_TOOBIG: "You can only select {0} item{1}",
		SELECT2_FREETAG_PREFIX: "Free text: ",

		DT_COLUMNS: "Columns"
	}

	MyAMS.plugins.i18n = {
		widgets: {},
		validate: {},
		datatables: {},
		fancybox: {
			ERROR: "Can't load requested content.",
			RETRY: "Please check URL or try again later.",
			CLOSE: "Close",
			NEXT: "Next",
			PREVIOUS: "Previous"
		},
		dndupload: {
			FILES_SELECTED: '{count} files selected',
			CHOOSE_FILE: 'Select file(s)',
			ADD_INFO: 'to add them to current folder,',
			DRAG_FILE: 'or drag and drop them here!',
			UPLOAD: 'Upload',
			UPLOADING: 'Uploading&hellip;',
			DONE: 'Done!',
			UPLOAD_MORE: 'Upload more?',
			ERROR: 'Error!',
			TRY_AGAIN: 'Try again?'
		}
	};


	$(document).ready(function() {
		$ = jQuery.noConflict();
		var html = $('HTML');
		html.removeClass('no-js')
			.addClass('js');
		var lang = html.attr('lang') || html.attr('xml:lang');
		if (lang && !lang.startsWith('en')) {
			MyAMS.lang = lang;
			MyAMS.getScript(MyAMS.baseURL + 'i18n/myams_' + lang.substr(0, 2) + MyAMS.devext + '.js', function () {
				MyAMS.initPage();
			});
		} else {
			MyAMS.initPage();
		}
	});

})(jQuery, this);
