/**
 * MyAMS callbacks management
 */
(function($, globals) {

	var MyAMS = globals.MyAMS,
		ams = MyAMS;

	MyAMS.callbacks = {

		/**
		 * Initialize list of callbacks
		 *
		 * Callbacks are initialized each time a page content is loaded and integrated into page's DOM.
		 * Unlike plug-ins, callbacks are called once in current's content context but are not kept into
		 * browser's memory for future use.
		 * Callbacks are defined via several data attributes:
		 * - data-ams-callback: name of function callback
		 * - data-ams-callback-source: source URL of file containing callback's function; can contain variables names
		 *   if enclosed between braces
		 * - data-ams-callback-options: JSON object containing callback options
		 */
		init: function(element) {
			$('[data-ams-callback]', element).each(function() {
				var self = this;
				var data = $(self).data();
				try {
					var callbacks = JSON.parse(data.amsCallback);
					if (!Array.isArray(callbacks)) {
						callbacks = [callbacks];
					}
				} catch (e) {
					callbacks = data.amsCallback.split(/\s+/);
				}
				for (var index=0; index < callbacks.length; index++) {
					var callback = callbacks[index];
					if (typeof(callback) === 'string') {
						var callback_func = ams.getFunctionByName(callback);
						var callback_options = data.amsCallbackOptions;
						if (typeof(callback_options) === 'string') {
							callback_options = callback_options.unserialize();
						}
						if (callback_func === undefined) {
							if (data.amsCallbackSource) {
								ams.getScript(data.amsCallbackSource,
									(function (cb) {
										ams.executeFunctionByName(cb, self, callback_options);
									})(callback), {
										async: data.amsCallbackAsync === undefined ? true : data.amsCallbackAsync
									});
							} else if (console) {
								console.warn && console.warn("Undefined callback: " + data.amsCallback);
							}
						} else {
							callback_func.call(self, callback_options);
						}
					} else {  // JSON object
						callback_func = ams.getFunctionByName(callback.callback);
						callback_options = callback.options;
						if (typeof(callback_options) === 'string') {
							callback_options = callback_options.unserialize();
						}
						if (callback_func === undefined) {
							if (callback.source) {
								ams.getScript(callback.source,
									(function (cb) {
										ams.executeFunctionByName(cb.callback, self, cb.options);
									})(callback), {
										async: callback.async === undefined ? true : callback.async
									});
							} else if (console) {
								console.warn && console.warn("Undefined callback: " + callback.callback);
							}
						} else {
							callback_func.call(self, callback.options);
						}
					}
				}
			});
		},

		/**
		 * Standard alert message callback
		 *
		 * An alert is an HTML div included on top of a "parent's" body
		 * Alert options include:
		 * - a status: 'info', 'warning', 'error' or 'success'
		 * - a parent: jQuery selector of parent's element
		 * - a header: alert's title
		 * - a subtitle
		 * - a message body
		 * - a boolean margin marker; if true, a 10 pixels margin will be added to alert's body
		 */
		alert: function(options) {
			var data = $(this).data();
			var settings = $.extend({}, options, data.amsAlertOptions);
			var parent = $(data.amsAlertParent || settings.parent || this);
			var status = data.amsAlertStatus || settings.status || 'info';
			var header = data.amsAlertHeader || settings.header;
			var message = data.amsAlertMessage || settings.message;
			var subtitle = data.amsAlertSubtitle || settings.subtitle;
			var margin = data.amsAlertMargin === undefined ? (settings.margin === undefined ? false : settings.margin) : data.amsAlertMargin;
			ams.skin.alert(parent, status, header, message, subtitle, margin);
		},

		/**
		 * Standard message box callback
		 *
		 * Message boxes are small informations messages displayed on bottom right page's corner
		 * Message box options include:
		 * - data-ams-messagebox-status: determines message box color; given as 'info', 'warning', 'error' or 'success'
		 * - data-ams-messagebox-title: message's title
		 * - data-ams-messagebox-content: message's HTML content
		 * - data-ams-messagebox-icon: if given, CSS class of message's icon
		 * - data-ams-messagebox-number: if given, a small error/message number displayed below message
		 * - data-ams-messagebox-timeout: if given, the message box will be automatically hidden passed this number
		 *   of milliseconds
		 * - data-ams-messagebox-callback: a callback's name, which will be called when message box is closed
		 */
		messageBox: function(options) {
			var data = $(this).data();
			var dataOptions = $.extend({}, options, data.amsMessageboxOptions);
			var settings = $.extend({}, dataOptions, {
				title: data.amsMessageboxTitle || dataOptions.title || '',
				content: data.amsMessageboxContent || dataOptions.content || '',
				icon: data.amsMessageboxIcon || dataOptions.icon,
				number: data.amsMessageboxNumber || dataOptions.number,
				timeout: data.amsMessageboxTimeout || dataOptions.timeout
			});
			var status = data.amsMessageboxStatus || dataOptions.status || 'info';
			var callback = ams.getFunctionByName(data.amsMessageboxCallback || dataOptions.callback);
			ams.skin.messageBox(status, settings, callback);
		},

		/**
		 * Standard small box callback
		 *
		 * Small boxes are notification messages displayed on top right page's corner.
		 * Small box options include:
		 * - data-ams-smallbox-status: determines message box color; given as 'info', 'warning', 'error' or 'success'
		 * - data-ams-smallbox-title: message's title
		 * - data-ams-smallbox-content: message's HTML content
		 * - data-ams-smallbox-icon: if given, CSS class of message's icon
		 * - data-ams-smallbox-icon-small: if given, CSS class of small message's icon
		 * - data-ams-smallbox-timeout: if given, the message box will be automatically hidden passed this number
		 *   of milliseconds
		 * - data-ams-smallbox-callback: a callback's name, which will be called when message box is closed
		 */
		smallBox: function(options) {
			var data = $(this).data();
			var dataOptions = $.extend({}, options, data.amsSmallboxOptions);
			var settings = $.extend({}, dataOptions, {
				title: data.amsSmallboxTitle || dataOptions.title || '',
				content: data.amsSmallboxContent || dataOptions.content || '',
				icon: data.amsSmallboxIcon || dataOptions.icon,
				iconSmall: data.amsSmallboxIconSmall || dataOptions.iconSmall,
				timeout: data.amsSmallboxTimeout || dataOptions.timeout
			});
			var status = data.amsSmallboxStatus || dataOptions.status || 'info';
			var callback = ams.getFunctionByName(data.amsSmallboxCallback || dataOptions.callback);
			ams.skin.smallBox(status, settings, callback);
		}
	};

})(jQuery, this);
