#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

# import standard library

# import interfaces
from pyams_skin.layer import IPyAMSLayer
from zope.component.interfaces import IObjectEvent, ObjectEvent

# import packages
from pyams_template.template import layout_config
from zope.interface import implementer, invariant, Interface, Attribute, Invalid
from zope.configuration.fields import GlobalInterface
from zope.schema import Text, TextLine, Choice, Int, Bool

from pyams_skin import _


class ISkin(Interface):
    """Skin interface

    Skins are registered as utilities implementing this interface
    and defining request layer as attribute
    """

    label = TextLine(title="Skin name")

    layer = GlobalInterface(title="Request layer",
                            description="This interface will be used to tag request layer",
                            required=True)


class ISkinChangedEvent(IObjectEvent):
    """Skin changed event"""


@implementer(ISkinChangedEvent)
class SkinChangedEvent(ObjectEvent):
    """Request skin changed event"""


class ISkinnable(Interface):
    """Skinnable content interface"""

    can_inherit_skin = Attribute("Check if skin can be inherited")

    inherit_skin = Bool(title=_("Inherit parent skin?"),
                        description=_("Should we reuse parent skin?"),
                        required=True,
                        default=False)

    no_inherit_skin = Bool(title=_("Don't inherit parent skin?"),
                           description=_("Should we override parent skin?"),
                           required=True,
                           default=True)

    skin_parent = Attribute("Skin parent (local or inherited)")

    skin = Choice(title=_("Custom graphic theme"),
                  description=_("This theme will be used to handle graphic design (colors and images)"),
                  vocabulary='PyAMS user skins',
                  required=False)

    @invariant
    def check_skin(self):
        if self.no_inherit_skin and not self.skin:
            raise Invalid(_("You must select a custom skin or inherit from parent!"))

    def get_skin(self, request=None):
        """Get skin matching this content"""


class IUserSkinnable(ISkinnable):
    """User skinnable content interface"""


@layout_config(template='templates/fullpage-layout.pt', layer=IPyAMSLayer)
class IFullPage(Interface):
    """Full page marker interface"""


@layout_config(template='templates/fullpage-modal-layout.pt', layer=IPyAMSLayer)
class IModalFullPage(IFullPage):
    """Full page modal dialog marker interface"""

    dialog_class = Attribute("Default dialog CSS class")


@layout_config(template='templates/inner-layout.pt', layer=IPyAMSLayer)
class IInnerPage(Interface):
    """Inner page marker interface"""


@layout_config(template='templates/widget-layout.pt', layer=IPyAMSLayer)
class IWidgetInnerPage(IInnerPage):
    """Inner page with widget marker interface"""


@layout_config(template='templates/modal-layout.pt', layer=IPyAMSLayer)
class IModalPage(Interface):
    """Modal page marker interface"""


class IDialog(IModalPage):
    """Modal dialog interface"""

    modal_class = TextLine(title="Modal dialog CSS class",
                           default='modal-medium')


class IPageHeader(Interface):
    """Page header interface used by 'header' content provider"""

    back_url = TextLine(title="Back URL",
                        required=False)

    back_target = TextLine(title="Back URL target",
                           description="HTML target selector, or None for full page target",
                           required=False)

    icon_class = TextLine(title='Icon CSS class')

    title = TextLine(title='Page title')

    title_badge = TextLine(title="Title badge")

    title_badge_class = TextLine(title="Title badge class")

    subtitle = TextLine(title='Page sub-title')

    subtitle_badge = TextLine(title="Sub-title badge")

    subtitle_badge_class = TextLine(title="Sub-title badge class")


class IContentHelp(Interface):
    """Content help block"""

    outer_margin = Int(title='Outer margin size',
                       default=0)

    status = TextLine(title='Help status',
                      default='info')

    header = TextLine(title='Help header')

    message = Text(title='Help message')

    message_format = Choice(title='Help message format',
                            vocabulary='PyAMS HTML renderers')


class IContentTitle(Interface):
    """Content title interface"""

    title = Attribute("Content title")


class IContextTitlePrefix(Interface):
    """Context title prefix interface"""

    prefix = Attribute("Context title prefix")


class IContentSearch(Interface):
    """Content search interface"""

    def get_search_results(self, data):
        """Extract search results from given data

        `data` is a dictionary containing search fields
        """


class ISearchPage(Interface):
    """Search page marker interface"""
