#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.dublincore.interfaces import IZopeDublinCore

from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.url import absolute_url


def get_json_refresh_event(object_id, content):
    """Get JSON response for basic refresh event"""
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshContent',
            'object_id': object_id,
            'content': content
        }
    }


def get_json_image_refresh_event(image, request, source=None):
    """Get JSON response for image refresh event"""
    dc = IZopeDublinCore(image)
    src = absolute_url(image if source is None else source, request)
    target = absolute_url(image, request)
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshImage',
            'src': src,
            'target': '{0}?_={1}'.format(target, dc.modified.timestamp())
        }
    }


def get_json_form_refresh_event(context, request, form_factory):
    """Get JSON response for form refresh event"""
    form = form_factory(context, request)
    form.ignoreRequest = True
    form.update()
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshContent',
            'object_id': form.id,
            'content': form.render()
        }
    }


def get_json_widget_refresh_event(context, request, form_factory, widget):
    """Get JSON response for form widget refresh event"""
    form = form_factory(context, request)
    form.ignoreRequest = True
    form.update()
    widget = form.widgets[widget]
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshWidget',
            'parent_id': form.id,
            'widget_name': widget.name,
            'content': widget.render()
        }
    }


def get_json_table_refresh_event(context, request, table_factory):
    """Get JSON response for table refresh event"""
    table = table_factory(context, request)
    table.update()
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshTable',
            'object_id': table.id,
            'table': table.render()
        }
    }


def get_json_switched_table_refresh_event(context, request, table_factory):
    """Get JSON response for switched table refresh event"""
    table = table_factory(context, request)
    table.update()
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshSwitchedTable',
            'object_id': table.id,
            'table': table.render()
        }
    }


def get_json_table_row_refresh_event(context, request, table_factory, table_item):
    """Get JSON response for table row refresh event"""
    table = table_factory(context, request)
    table.update()
    row = table.setUpRow(table_item)
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshRow',
            'object_id': '{0}::{1}'.format(table.id, ICacheKeyValue(table_item)),
            'row': table.renderRow(row)
        }
    }


def get_json_table_cell_refresh_event(context, request, object_id, col_name, cell_content=None, column_factory=None):
    """Get JSON response for table cell refresh event"""
    if cell_content is None:
        column = column_factory(context, request, None)
        cell_content = column.renderCell(context)
    return {
        'event': 'myams.refresh',
        'options': {
            'handler': 'MyAMS.skin.refreshRowCell',
            'object_id': object_id,
            'col_name': col_name,
            'cell': cell_content
        }
    }
