#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_skin.interfaces.container import IContainerBaseView
from pyams_skin.layer import IPyAMSLayer
from zope.container.interfaces import IContainer

# import packages
from pyams_form.security import get_edit_permission
from pyams_skin.table import BaseTable
from pyams_template.template import template_config
from pyramid.exceptions import NotFound
from pyramid.httpexceptions import HTTPUnauthorized, HTTPInternalServerError
from pyramid.view import view_config
from zope.interface import implementer

from pyams_utils import _


@template_config(template='templates/container.pt', layer=IPyAMSLayer)
@implementer(IContainerBaseView)
class ContainerView(object):
    """Base container view"""

    table_class = BaseTable

    def __init__(self, context, request):
        super(ContainerView, self).__init__(context, request)
        self.table = self.table_class(context, request)

    def update(self):
        super(ContainerView, self).update()
        self.table.update()


@view_config(name='delete-element.json', context=IContainer, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def delete_container_element(request, container_factory=None, ignore_permission=False):
    """Delete container element

    This view is not strictly protected, but:
    - either the function is called from another protected view
    - either the view is checking edit permission from context adapter; if permission can't be found,
      an internal server error is raised!
    If the function is called from another unprotected view with 'ignore_permission=True',
    it's a configuration error.
    """
    translate = request.localizer.translate
    # Get object name to be removed
    name = request.params.get('object_name')
    if not name:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("No provided object_name argument!"))
            }
        }
    # Check container factory
    container = request.context
    if container_factory is not None:
        container = container_factory(container)
    # Check container
    if name not in container:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("Given element name doesn't exist!"))
            }
        }
    # Check permission
    if not ignore_permission:
        context = container[name]
        permission = get_edit_permission(request, context)
        if permission is None:
            raise HTTPInternalServerError("Missing permission definition")
        elif not request.has_permission(permission, context):
            raise HTTPUnauthorized()
    # Delete element
    del container[name]
    return {'status': 'success'}


def switch_element_attribute(request, interface, adapter_name='', attribute_name=''):
    """Sswitch container element attribute

    :param request: original browser request; request should contain a parameter called
        "object_name" which contains the name of the element which should be switched.
        A NotFound exception is raised if argument is not provided or if given argument
        doesn't match an existing element.
    :param interface: container interface to which request's context should be adapted
    :param adapter_name: name of the adapter to be used to get given interface
    :param attribute_name: name of the boolean attribute to be switched
    :return: a JSON object containing a boolean "attribute" property defining new element value.
    """
    context = request.context
    if interface.providedBy(context):
        container = context
    else:
        container = request.registry.queryAdapter(context, interface, name=adapter_name)
        if container is None:
            raise NotFound()
    object_name = request.params.get('object_name')
    if not object_name:
        raise NotFound()
    element = container.get(str(object_name))
    if element is None:
        raise NotFound()
    setattr(element, attribute_name, not getattr(element, attribute_name))
    return {attribute_name: getattr(element, attribute_name)}


def switch_element_visibility(request, interface, adapter_name=''):
    """Set container element visibility

    :param request: original browser request; request should contain a parameter called
        "object_name" which contains the name of the element which should be switched.
        A NotFound exception is raised if argument is not provided or if given argument
        doesn't match an existing element.
    :param interface: container interface to which request's context should be adapted
    :return: a JSON object containing a boolean "visible" property defining new element visibility.
    """
    return switch_element_attribute(request, interface, adapter_name, 'visible')
