#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from z3c.form.browser.object import ObjectWidget
from z3c.form.interfaces import IFieldWidget, IObjectFactory
from z3c.form.object import getIfName
from z3c.form.widget import FieldWidget
from zope.interface import Interface, alsoProvides, implementer_only

from pyams_form.interfaces.form import IForm, IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_gis.interfaces import IGeoPointZ
from pyams_gis.interfaces.widget import IGeoPointWidget, IGeoPointZWidget
from pyams_gis.point import GeoPoint, GeoPointZ
from pyams_gis.schema import IGeoPoint, IGeoPointField, IGeoPointZField
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData


__docformat__ = 'restructuredtext'


@adapter_config(name=getIfName(IGeoPoint),
                context=(Interface, IFormLayer, IForm, IGeoPointWidget),
                provides=IObjectFactory)
def geo_point_object_widget_factory(*args, **kwargs):
    """GeoPointZ object factory"""
    return GeoPoint


@widgettemplate_config(mode='input',
                       template='templates/geopoint-input.pt', layer=IFormLayer)
@implementer_only(IGeoPointWidget)
class GeoPointWidget(ObjectWidget):
    """GeoPoint widget"""

    def updateWidgets(self, setErrors=True):
        super(GeoPointWidget, self).updateWidgets(setErrors)
        widgets = self.widgets
        longitude = widgets['longitude']
        longitude.readonly = 'readonly'
        longitude.label_css_class = 'control-label col-md-3'
        longitude.input_css_class = 'col-md-2'
        longitude.object_data = {
            'ams-change-handler': 'PyAMS_GIS.position.changedCoordinate'
        }
        alsoProvides(longitude, IObjectData)
        latitude = widgets['latitude']
        latitude.readonly = 'readonly'
        latitude.label_css_class = 'control-label col-md-3'
        latitude.input_css_class = 'col-md-2'
        latitude.object_data = {
            'ams-change-handler': 'PyAMS_GIS.position.changedCoordinate'
        }
        alsoProvides(latitude, IObjectData)
        projection = widgets['projection']
        projection.label_css_class = 'control-label col-md-3'
        projection.input_css_class = 'col-md-9'
        projection.object_data = {
            'ams-events-handlers': {
                'change.select2': 'PyAMS_GIS.position.changedProjection'
            }
        }
        alsoProvides(projection, IObjectData)

    @property
    def wgs_coordinates(self):
        value = self.field.get(self.field.interface(self.context))
        if not value:
            return json.dumps({
                'longitude': None,
                'latitude': None
            })
        # else:
        point = value.wgs_coordinates
        return json.dumps({
            'longitude': float(point[0]),
            'latitude': float(point[1])
        })


@adapter_config(context=(IGeoPointField, IFormLayer),
                provides=IFieldWidget)
def GeoPointFieldWidget(field, request):
    """GeoPoint field widget factory"""
    return FieldWidget(field, GeoPointWidget(request))


@adapter_config(name=getIfName(IGeoPointZ),
                context=(Interface, IFormLayer, IForm, IGeoPointZWidget),
                provides=IObjectFactory)
def geo_pointz_object_factory(*args, **kwargs):
    return GeoPointZ


@widgettemplate_config(mode='input',
                       template='templates/geopoint-input.pt', layer=IFormLayer)
@implementer_only(IGeoPointZWidget)
class GeoPointZWidget(ObjectWidget):
    """GeoPointZ widget"""

    def updateWidgets(self, setErrors=True):
        super(GeoPointZWidget, self).updateWidgets(setErrors)
        widgets = self.widgets
        longitude = widgets['longitude']
        longitude.label_css_class = 'control-label col-md-3'
        longitude.input_css_class = 'col-md-2'
        longitude.object_data = {
            'ams-change-handler': 'PyAMS_GIS.position.changedCoordinate'
        }
        alsoProvides(longitude, IObjectData)
        latitude = widgets['latitude']
        latitude.label_css_class = 'control-label col-md-3'
        latitude.input_css_class = 'col-md-2'
        latitude.object_data = {
            'ams-change-handler': 'PyAMS_GIS.position.changedCoordinate'
        }
        alsoProvides(latitude, IObjectData)
        projection = widgets['projection']
        projection.label_css_class = 'control-label col-md-3'
        latitude.input_css_class = 'col-md-9'
        projection.object_data = {
            'ams-events-handlers': {
                'change.select2': 'PyAMS_GIS.position.changedProjection'
            }
        }
        alsoProvides(projection, IObjectData)
        altitude = widgets['altitude']
        altitude.label_css_class = 'control-label col-md-3'
        altitude.input_css_class = 'col-md-2'

    @property
    def wgs_coordinates(self):
        value = self.field.get(self.field.interface(self.context))
        if not value:
            return json.dumps({
                'longitude': None,
                'latitude': None
            })
        # else:
        point = value.wgs_coordinates
        return json.dumps({
            'longitude': float(point[0]),
            'latitude': float(point[1])
        })


@adapter_config(context=(IGeoPointZField, IFormLayer),
                provides=IFieldWidget)
def GeoPointZFieldWidget(field, request):
    """GeoPointZ field widget factory"""
    return FieldWidget(field, GeoPointZWidget(request))
