#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from z3c.form.browser.object import ObjectWidget
from z3c.form.interfaces import IFieldWidget, IObjectFactory
from z3c.form.object import getIfName
from z3c.form.widget import FieldWidget
from zope.interface import Interface, alsoProvides, implementer_only

from pyams_form.interfaces.form import IForm, IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_gis.area import GeoArea
from pyams_gis.interfaces import IGeoArea
from pyams_gis.interfaces.widget import IGeoAreaWidget
from pyams_gis.schema import IGeoAreaField
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData


__docformat__ = 'restructuredtext'


@adapter_config(name=getIfName(IGeoArea),
                context=(Interface, IFormLayer, IForm, IGeoAreaWidget),
                provides=IObjectFactory)
def geo_area_object_factory(*args, **kwargs):
    """GeoArea object factory"""
    return GeoArea


@widgettemplate_config(mode='input',
                       template='templates/geoarea-input.pt', layer=IFormLayer)
@implementer_only(IGeoAreaWidget)
class GeoAreaWidget(ObjectWidget):
    """GeoArea widget"""

    def updateWidgets(self, setErrors=True):
        super(GeoAreaWidget, self).updateWidgets()
        widgets = self.widgets
        x1 = widgets['x1']
        x1.readonly = 'readonly'
        x1.input_css_class = 'col-md-2'
        x1.object_data = {
            'ams-change-handler': 'PyAMS_GIS.area.changedCoordinate'
        }
        alsoProvides(x1, IObjectData)
        x2 = widgets['x2']
        x2.readonly = 'readonly'
        x2.input_css_class = 'col-md-2'
        x2.object_data = {
            'ams-change-handler': 'PyAMS_GIS.area.changedCoordinate'
        }
        alsoProvides(x2, IObjectData)
        y1 = widgets['y1']
        y1.readonly = 'readonly'
        y1.input_css_class = 'col-md-2'
        y1.object_data = {
            'ams-change-handler': 'PyAMS_GIS.area.changedCoordinate'
        }
        alsoProvides(y1, IObjectData)
        y2 = widgets['y2']
        y2.readonly = 'readonly'
        y2.input_css_class = 'col-md-2'
        y2.object_data = {
            'ams-change-handler': 'PyAMS_GIS.area.changedCoordinate'
        }
        alsoProvides(y2, IObjectData)
        projection = widgets['projection']
        projection.object_data = {
            'ams-events-handlers': {
                'change.select2': 'PyAMS_GIS.area.changedProjection'
            }
        }
        alsoProvides(projection, IObjectData)

    @property
    def wgs_coordinates(self):
        value = self.field.get(self.field.interface(self.context))
        if not value:
            return json.dumps({
                'x1': None,
                'y1': None,
                'x2': None,
                'y2': None
            })
        else:
            point1, point2 = value.wgs_coordinates
            return json.dumps({
                'x1': float(point1[0]),
                'y1': float(point1[1]),
                'x2': float(point2[0]),
                'y2': float(point2[1])
            })


@adapter_config(context=(IGeoAreaField, IFormLayer),
                provides=IFieldWidget)
def GeoAreaFieldWidget(field, request):
    """GeoArea field widget factory"""
    return FieldWidget(field, GeoAreaWidget(request))
