#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_form.form import AJAXAddForm, AJAXEditForm
from pyams_gis.interfaces.layer import ITileMapLayer, IWMSMapLayer, IGeoportalMapLayer, IEsriFeatureMapLayer, \
    IGoogleMapLayer
from pyams_gis.interfaces.utility import IMapManager
from pyams_gis.layer import TileMapLayer, WMSMapLayer, GeoportalMapLayer, EsriFeatureMapLayer, GoogleMapLayer
from pyams_gis.zmi.interfaces import ILayerAddForm
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces import MANAGE_SYSTEM_PERMISSION
from pyams_zmi.layer import IAdminLayer
from z3c.form.interfaces import IDataExtractedEvent, DISPLAY_MODE
from zope.component.interfaces import ISite

# import packages
from pyams_gis.zmi.utility import MapManagerLayersTable
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.registry import query_utility
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form import field
from zope.interface import implementer, Invalid

from pyams_gis import _


@subscriber(IDataExtractedEvent, form_selector=ILayerAddForm)
def handle_new_geoportal_layer_data(event):
    """Handle new layer data extraction"""
    manager = query_utility(IMapManager)
    name = event.data.get('name')
    if name in manager:
        event.form.widgets.errors += (Invalid(_("Specified layer name already exists!")))


#
# Base tiles layer forms
#

@viewlet_config(name='add-tile-layer.menu', context=ISite, layer=IAdminLayer,
                view=MapManagerLayersTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=10)
class TileLayerAddMenu(ToolbarMenuItem):
    """Tile layer add menu"""

    label = _("Add tile layer...")
    label_css_class = 'fa fa-fw fa-globe'
    url = 'add-tile-layer.html'
    modal_target = True


@pagelet_config(name='add-tile-layer.html', context=ISite, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@implementer(ILayerAddForm)
class TileLayerAddForm(AdminDialogAddForm):
    """Tile layer add form"""

    title = _("Maps manager")
    legend = _("Add tile layer")
    icon_css_class = 'fa fa-fw fa-globe'
    dialog_class = 'modal-large'

    fields = field.Fields(ITileMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'add-tile-layer.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def create(self, data):
        return TileMapLayer()

    def add(self, layer):
        context = query_utility(IMapManager)
        context[layer.name] = layer

    def nextURL(self):
        return absolute_url(self.context, self.request, 'map-manager.html')


@view_config(name='add-tile-layer.json', context=ISite, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class TileLayerAJAXAddForm(AJAXAddForm, TileLayerAddForm):
    """Tile layer add form, JSON renderer"""


@pagelet_config(name='properties.html', context=ITileMapLayer, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
class TileLayerEditForm(AdminDialogEditForm):
    """Tile layer edit form"""

    prefix = 'tile_properties.'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Edit layer properties")
    icon_css_class = 'fa fa-fw fa-globe'
    dialog_class = 'modal-large'

    fields = field.Fields(ITileMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def updateWidgets(self, prefix=None):
        super(TileLayerEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE


@view_config(name='properties.json', context=ITileMapLayer, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class TileLayerAJAXEditForm(AJAXEditForm, TileLayerEditForm):
    """Tile layer edit form, JSON renderer"""


#
# WMS layer forms
#

@viewlet_config(name='add-wms-layer.menu', context=ISite, layer=IAdminLayer,
                view=MapManagerLayersTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=20)
class WMSLayerAddMenu(ToolbarMenuItem):
    """WMS layer add menu"""

    label = _("Add WMS layer...")
    label_css_class = 'fa fa-fw fa-map'
    url = 'add-wms-layer.html'
    modal_target = True


@pagelet_config(name='add-wms-layer.html', context=ISite, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@implementer(ILayerAddForm)
class WMSLayerAddForm(AdminDialogAddForm):
    """WMS layer add form"""

    title = _("Maps manager")
    legend = _("Add WMS layer")
    icon_css_class = 'fa fa-fw fa-map'
    dialog_class = 'modal-large'

    fields = field.Fields(IWMSMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'add-wms-layer.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def create(self, data):
        return WMSMapLayer()

    def add(self, layer):
        context = query_utility(IMapManager)
        context[layer.name] = layer

    def nextURL(self):
        return absolute_url(self.context, self.request, 'map-manager.html')


@view_config(name='add-wms-layer.json', context=ISite, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class WMSLayerAJAXAddForm(AJAXAddForm, WMSLayerAddForm):
    """WMS layer add form, JSON renderer"""


@pagelet_config(name='properties.html', context=IWMSMapLayer, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
class WMSLayerEditForm(AdminDialogEditForm):
    """WMS layer edit form"""

    prefix = 'wms_properties.'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Edit layer properties")
    icon_css_class = 'fa fa-fw fa-map'
    dialog_class = 'modal-large'

    fields = field.Fields(IWMSMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def updateWidgets(self, prefix=None):
        super(WMSLayerEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE


@view_config(name='properties.json', context=IWMSMapLayer, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class WMSLayerAJAXEditForm(AJAXEditForm, WMSLayerEditForm):
    """WMS layer edit form, JSON renderer"""


#
# Geoportal layer forms
#

@viewlet_config(name='add-geoportal-layer.menu', context=ISite, layer=IAdminLayer,
                view=MapManagerLayersTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=50)
class GeoportalLayerAddMenu(ToolbarMenuItem):
    """Geoportal layer add menu"""

    label = _("Add Geoportal WMS layer...")
    label_css_class = 'fa fa-fw fa-map-o'
    url = 'add-geoportal-layer.html'
    modal_target = True


@pagelet_config(name='add-geoportal-layer.html', context=ISite, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@implementer(ILayerAddForm)
class GeoportalLayerAddForm(AdminDialogAddForm):
    """Geoportal layer add form"""

    title = _("Maps manager")
    legend = _("Add IGN Geoportal layer")
    icon_css_class = 'fa fa-fw fa-map-o'
    dialog_class = 'modal-large'

    fields = field.Fields(IGeoportalMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'add-geoportal-layer.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def create(self, data):
        return GeoportalMapLayer()

    def add(self, layer):
        context = query_utility(IMapManager)
        context[layer.name] = layer

    def nextURL(self):
        return absolute_url(self.context, self.request, 'map-manager.html')


@view_config(name='add-geoportal-layer.json', context=ISite, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class GeoportalLayerAJAXAddForm(AJAXAddForm, GeoportalLayerAddForm):
    """Geoportal layer add form, JSON renderer"""


@pagelet_config(name='properties.html', context=IGeoportalMapLayer, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
class GeoportalLayerEditForm(AdminDialogEditForm):
    """Geoportal layer edit form"""

    prefix = 'geoportal_properties.'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Edit layer properties")
    icon_css_class = 'fa fa-fw fa-map-o'
    dialog_class = 'modal-large'

    fields = field.Fields(IGeoportalMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def updateWidgets(self, prefix=None):
        super(GeoportalLayerEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE


@view_config(name='properties.json', context=IGeoportalMapLayer, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class GeoportalLayerAJAXEditForm(AJAXEditForm, GeoportalLayerEditForm):
    """Geoportal layer edit form, JSON renderer"""


#
# ESRI feature layer forms
#

@viewlet_config(name='add-esri-feature-layer.menu', context=ISite, layer=IAdminLayer,
                view=MapManagerLayersTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=70)
class EsriFeatureLayerAddMenu(ToolbarMenuItem):
    """ESRI feature layer add menu"""

    label = _("Add ESRI feature layer...")
    label_css_class = 'fa fa-fw fa-map-signs'
    url = 'add-esri-feature-layer.html'
    modal_target = True


@pagelet_config(name='add-esri-feature-layer.html', context=ISite, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@implementer(ILayerAddForm)
class EsriFeatureLayerAddForm(AdminDialogAddForm):
    """ESRI feature layer add form"""

    title = _("Maps manager")
    legend = _("Add ESRI feature layer")
    icon_css_class = 'fa fa-fw fa-map-signs'
    dialog_class = 'modal-large'

    fields = field.Fields(IEsriFeatureMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'add-esri-feature-layer.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def create(self, data):
        return EsriFeatureMapLayer()

    def add(self, layer):
        context = query_utility(IMapManager)
        context[layer.name] = layer

    def nextURL(self):
        return absolute_url(self.context, self.request, 'map-manager.html')


@view_config(name='add-esri-feature-layer.json', context=ISite, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class EsriFeatureLayerAJAXAddForm(AJAXAddForm, EsriFeatureLayerAddForm):
    """ESRI feature layer add form, JSON renderer"""


@pagelet_config(name='properties.html', context=IEsriFeatureMapLayer, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
class EsriFeatureLayerEditForm(AdminDialogEditForm):
    """ESRI feature layer edit form"""

    prefix = 'esri_properties.'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Edit layer properties")
    icon_css_class = 'fa fa-fw fa-map-signs'
    dialog_class = 'modal-large'

    fields = field.Fields(IEsriFeatureMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def updateWidgets(self, prefix=None):
        super(EsriFeatureLayerEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE


@view_config(name='properties.json', context=IEsriFeatureMapLayer, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class EsriFeatureLayerAJAXEditForm(AJAXEditForm, EsriFeatureLayerEditForm):
    """ESRI feature layer edit form, JSON renderer"""


#
# Google maps layer forms
#

@viewlet_config(name='add-google-layer.menu', context=ISite, layer=IAdminLayer,
                view=MapManagerLayersTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=80)
class GoogleMapLayerAddMenu(ToolbarMenuItem):
    """Google Maps layer add menu"""

    label = _("Add Google Maps layer...")
    label_css_class = 'fa fa-fw fa-map-marker'
    url = 'add-google-layer.html'
    modal_target = True


@pagelet_config(name='add-google-layer.html', context=ISite, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
@implementer(ILayerAddForm)
class GoogleMapLayerAddForm(AdminDialogAddForm):
    """Google Maps layer add form"""

    title = _("Maps manager")
    legend = _("Add Google Maps layer")
    icon_css_class = 'fa fa-fw fa-map-marker'
    dialog_class = 'modal-large'

    fields = field.Fields(IGoogleMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'add-google-layer.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def create(self, data):
        return GoogleMapLayer()

    def add(self, layer):
        context = query_utility(IMapManager)
        context[layer.name] = layer

    def nextURL(self):
        return absolute_url(self.context, self.request, 'map-manager.html')


@view_config(name='add-google-layer.json', context=ISite, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class GoogleMapLayerAJAXAddForm(AJAXAddForm, GoogleMapLayerAddForm):
    """Google Maps layer add form, JSON renderer"""


@pagelet_config(name='properties.html', context=IGoogleMapLayer, layer=IPyAMSLayer,
                permission=MANAGE_SYSTEM_PERMISSION)
class GoogleMapLayerEditForm(AdminDialogEditForm):
    """Google Maps layer edit form"""

    prefix = 'google_properties.'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Edit layer properties")
    icon_css_class = 'fa fa-fw fa-map-marker'
    dialog_class = 'modal-large'

    fields = field.Fields(IGoogleMapLayer).omit('__name__', '__parent__')
    ajax_handler = 'properties.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def updateWidgets(self, prefix=None):
        super(GoogleMapLayerEditForm, self).updateWidgets(prefix)
        if 'name' in self.widgets:
            self.widgets['name'].mode = DISPLAY_MODE


@view_config(name='properties.json', context=IGoogleMapLayer, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class GoogleMapLayerAJAXEditForm(AJAXEditForm, GoogleMapLayerEditForm):
    """Google Maps layer edit form, JSON renderer"""
