#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library
try:
    from osgeo.osr import SpatialReference, CoordinateTransformation
    have_gdal = True
except ImportError:
    have_gdal = False

# import interfaces

# import packages
from pyams_gis.transform import transform
from pyramid_rpc.jsonrpc import jsonrpc_method


@jsonrpc_method(endpoint='gis')
def transformPoint(request, point, from_srid, to_srid):
    """Transform point given in source projection to another projection

    :param point: point coordinates given as a mapping with 'longitude' and 'latitude' values
    :param from_srid: source coordinates system given as SRID
    :param to_srid: target coordinates system given as SRID
    :return: mapping with new 'point' key containing transformed coordinates, and 'projection'
        key containing SRID of result projection system
    """
    return transform(point, from_srid, to_srid)


@jsonrpc_method(endpoint='gis')
def transformArea(request, area, from_srid, to_srid):
    """Transform area given in source projection to another projection

    :param area: area coordinates given as a mapping with 'x1', 'y1', 'x2', and 'y2' values
    :param from_srid: source coordinates system given as SRID
    :param to_srid: target coordinates system given as SRID
    :return: mapping with new 'area' key containing transformed coordinates, and 'projection'
        key containing SRID of result projection system
    """
    from_srid = int(from_srid)
    to_srid = int(to_srid)
    if (not have_gdal) or (from_srid == to_srid):
        return {
            'area': area,
            'srid': from_srid
        }
    point1 = transform((area['x1'], area['y1']), from_srid, to_srid)
    point2 = transform((area['x2'], area['y2']), from_srid, to_srid)
    return {
        'area': {
            'x1': point1['point']['longitude'],
            'y1': point1['point']['latitude'],
            'x2': point2['point']['longitude'],
            'y2': point2['point']['latitude']
        },
        'srid': point1['srid']
    }
