/**
 * PyAMS_GIS package extension
 */
(function($, globals) {

	"use strict";

	var MyAMS = globals.MyAMS;
	var L;

	var PyAMS_GIS = {

		RPC_ENDPOINT: '/api/gis/json',
		WGS_SRID: 4326,

		_layersControlAddItem: function(obj) {
			var group = $('<div></div>').addClass('inline-group'),
				label = $('<label></label>').addClass(obj.overlay ? "checkbox" : "radio"),
				input,
				span = $('<i></i>'),
				name,
				checked = this._map.hasLayer(obj.layer);
			if (obj.overlay) {
				input = document.createElement('input');
				input.type = 'checkbox';
				input.className = 'leaflet-control-layers-selector';
				input.defaultChecked = checked;
			} else {
				input = this._createRadioElement('leaflet-base-layers', checked);
			}
			input.layerId = L.stamp(obj.layer);
			$(input).addClass(obj.overlay ? "checkbox" : "radio");
			L.DomEvent.on(input, 'click', this._onInputClick, this);
			name = $('<span></span>').text(' ' + obj.name);
			label.append(input);
			label.append(span);
			label.append(name);
			group.append(label);
			var container = obj.overlay ? this._overlaysList : this._baseLayersList;
			$(container).append(group);
			// $($(container).parents('form').get(0)).data("async", true);
			return group;
		},

		init: function(context, options, callback, globalCallback) {
			MyAMS.ajax.check([
					globals.L
				], [
					'/--static--/pyams_gis/js/leaflet-1.7.1' + MyAMS.devext + '.js'
				], function(first_load) {
					var required = [];
					if (first_load) {
						L = globals.L;
						L.Control.Layers.prototype._addItem = PyAMS_GIS._layersControlAddItem;
						required.push(MyAMS.getScript(
							'/--static--/pyams_gis/js/leaflet-gesture-handling-1.2.1' + MyAMS.devext + '.js'));
						required.push(MyAMS.getScript(
							'/--static--/pyams_gis/js/leaflet-fullscreen' + MyAMS.devext + '.js'));
						MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-1.7.1' + MyAMS.devext + '.css',
							'leaflet');
						MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-gesture-handling-1.2.1' + MyAMS.devext + '.css',
							'leaflet-gesture-handling');
						MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-fullscreen' + MyAMS.devext + '.css',
							'leaflet-fullscreen');
					}
					$.when.apply($, required).then(function() {

						function createMap(map, config) {
							var data = map.data(),
								settings = {
									preferCanvas: data.mapLeafletPreferCanvas || false,
									attributionControl: data.mapLeafletAttributionControl === undefined ?
										config.attributionControl :
										data.mapLeafletAttributionControl,
									zoomControl: data.mapLeafletZoomControl === undefined ?
										config.zoomControl :
										data.mapLeafletZoomControl,
									fullscreenControl: data.mapLeafletFullscreen === undefined ?
										config.fullscreenControl && {
											pseudoFullscreen: true
										} || null :
										data.mapLeafletFullscreen,
									crs: data.mapLeafletCrs || MyAMS.getObject(config.crs) || globals.L.CRS.EPSG3857,
									center: data.mapLeafletCenter || config.center,
									zoom: data.mapLeafletZoom || config.zoom,
									gestureHandling: data.mapLeafletWheelZoom === undefined ?
										!config.scrollWheelZoom :
										data.mapLeafletWheelZoom,
									keyboard: data.mapLeafletKeyboard === undefined ?
										config.keyboard && !L.Browser.mobile :
										data.amsLeafletKeyboard
								};
							settings = $.extend({}, settings, options);
							map.trigger('map.init', [map, settings, config]);
							var leafmap = L.map(map.attr('id'), settings);
							var layersConfig = [];
							if (config.layers) {
								for (var idx = 0; idx < config.layers.length; idx++) {
									var layerConfig = config.layers[idx];
									map.trigger('map.layer.init', [map, layerConfig]);
									layersConfig.push(PyAMS_GIS.getLayer(map, leafmap, layerConfig));
								}
							} else {
								layersConfig.push(L.tileLayer('http://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
									maxZoom: 19,
									id: 'osm',
									attribution: '&copy; <a href="http://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
								}));
							}
							$.when.apply($, layersConfig).then(function(...layers) {
								for (var idx=0; idx < layers.length; idx++) {
									layers[idx].addTo(leafmap);
								}
								if (config.zoomControl && (data.mapLeafletHideZoomControl !== true)) {
									L.control.scale().addTo(leafmap);
								}
								if (config.center) {
									leafmap.setView(new L.LatLng(config.center.lat, config.center.lon),
													config.zoom || 13);
								} else if (config.bounds) {
									leafmap.fitBounds(config.bounds);
								}
								map.data('leafmap', leafmap);
								map.data('leafmap.config', config);
								map.data('leafmap.layers', layers.reduce((res, layer) => ({...res, [layer.options.name]: layer}), {}));
								map.trigger('map.finishing', [map, leafmap, config]);
								if (callback) {
									callback(leafmap, config);
								}
								map.trigger('map.finished', [map, leafmap, config]);
							});
						}

						var maps = $.map(context, function(elt) {
							return new Promise(function(resolve, reject) {
								var map = $(elt),
									data = map.data(),
									config = data.mapConfiguration;
								if (config) {
									createMap(map, config);
									resolve();
								} else {
									MyAMS.ajax.post(data.mapConfigurationUrl || 'get-map-configuration.json', {}, function(config) {
										createMap(map, config);
										resolve();
									});
								}
							});
						});
						if (globalCallback) {
							$.when.apply($, maps).then(globalCallback);
						}
					});
				}
			);
		},

		getLayer: function(map, leafmap, layer) {
			var factory = MyAMS.getObject(layer.factory);
			if (factory !== undefined) {
				delete layer.factory;
				var deferred = [];
				if (layer.dependsOn) {
					for (var name in layer.dependsOn) {
						if (!layer.dependsOn.hasOwnProperty(name)) {
							continue;
						}
						if (MyAMS.getObject(name) === undefined) {
							deferred.push(MyAMS.getScript(layer.dependsOn[name]));
						}
					}
					delete layer.dependsOn;
				}
				if (deferred.length > 0) {
					$.when.apply($, deferred);
				}
				return factory(map, leafmap, layer);
			}
		},

		factory: {
			GeoJSON: function(map, leafmap, layer) {
				var url = layer.url;
				delete layer.url;
				var result = L.geoJSON(null, layer);
				map.on('map.finished', function(evt, map, leafmap, config) {
					$.get(url, function(data) {
						result.addData(data.geometry, {
							style: layer.style
						});
						if (config.fitLayer === layer.name) {
							leafmap.fitBounds(result.getBounds());
						}
					});
				});
				return result;
			},
			TileLayer: function(map, leafmap, layer) {
				var url = layer.url;
				delete layer.url;
				return L.tileLayer(url, layer);
			},
			WMS: function(map, leafmap, layer) {
				var url = layer.url;
				delete layer.url;
				return L.tileLayer.wms(url, layer);
			},
			Geoportal: {
				WMS: function(map, leafmap, layer) {
					MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-gp-3.0.2' + MyAMS.devext + '.css', 'geoportal');
					return L.geoportalLayer.WMS(layer);
				}
			},
			ESRI: {
				Feature: function(map, leafmap, layer) {
					return L.esri.featureLayer(layer);
				}
			},
			Google: function(map, leafmap, layer) {
				var apiKey = layer.apiKey;
				delete layer.apiKey;
				if (MyAMS.getObject('window.google.maps') === undefined) {
					var script = MyAMS.getScript('https://maps.googleapis.com/maps/api/js?key=' + apiKey);
					$.when.apply($, [script]);
				}
				return L.gridLayer.googleMutant(layer);
			}
		},

		callJSON: function(method, params, callback) {
			MyAMS.ajax.check($.jsonRPC,
							 MyAMS.baseURL + 'ext/jquery-jsonrpc' + MyAMS.devext + '.js',
							 function() {
								$.jsonRPC.withOptions({
									endPoint: PyAMS_GIS.RPC_ENDPOINT,
									cache: false
								}, function() {
									$.jsonRPC.request(method, {
										id: new Date().getTime(),
										params: params,
										success: callback,
										error: MyAMS.error.show
									});
								});
							 });
		},

		/**
		 * Init markers layer
		 */
		markers: {

			init: function(maps, callback) {

				MyAMS.ajax.check([
					L.MarkerClusterGroup
				], [
					'/--static--/pyams_gis/js/leaflet-markercluster-1.4.1' + MyAMS.devext + '.js'
				], function(firstLoad) {

					if (firstLoad) {
						MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-markercluster-1.4.1' + MyAMS.devext + '.css',
							'leaflet-markercluster');
						MyAMS.getCSS('/--static--/pyams_gis/css/leaflet-markercluster-default-1.4.1' + MyAMS.devext + '.css',
							'leaflet-markercluster-default');
					}

					maps.each(function(idx, elt) {

						var map = $(elt),
							data = map.data(),
							leafmap = map.data('leafmap'),
							config = map.data('leafmap.config'),
							markers = data.mapMarkers;

						var markerIcon,
							activeIcon;

						// show tooltip
						var hoverMarker = function(evt) {
							this.setIcon(activeIcon);
							this.openPopup();
							var marker = $(`[id="marker-${this.options.markerId}"]`);
							if (marker.exists()) {
								marker.addClass(marker.data('ams-active-class') || 'active');
								if (evt.scroll !== false) {
									MyAMS.ajax.check($.scrollTo,
										MyAMS.baseURL + 'ext/jquery-scrollto-2.1.2' + MyAMS.devext + '.js',
										function() {
											marker.parents('.markers-target').scrollTo(marker);
										});
								}
							}
						}
						// hide tooltip
						var leaveMarker = function(e) {
							this.closePopup();
							this.setIcon(markerIcon);
							var marker = $(`[id="marker-${this.options.markerId}"]`);
							if (marker.exists()) {
								marker.removeClass(marker.data('ams-active-class') || 'active');
							}
						}
						// click marker
						var clickMarker = function(e) {
							if (data.mapLinksTarget) {
								window.open(this.options.clickURL, data.mapLinksTarget);
							} else {
								window.location.href = this.options.clickURL;
							}
						}

						if (markers) {
							// create custom icon
							markerIcon = L.icon({
								iconUrl: markers.icon.url,
								iconSize: markers.icon.size,
								iconAnchor: markers.icon.anchor
							});
							var activeIconSize = [
								markers.icon.size[0] * 1.25,
								markers.icon.size[1] * 1.25
							];
							activeIcon = L.icon({
								iconUrl: markers.icon.url,
								iconSize: activeIconSize,
								iconAnchor: [
									activeIconSize[0] / 2,
									activeIconSize[1] - 1
								]
							});
							// customize cluster icon
							var markersClusterCustom;
							if (data.mapMarkersClusters === false) {
								markersClusterCustom = L.featureGroup();
							} else {
								markersClusterCustom = L.markerClusterGroup({
									iconCreateFunction: function(cluster) {
										return L.divIcon({
											html: cluster.getChildCount(),
											className: markers.clusterClass || 'map-cluster',
											iconSize: null
										});
									}
								});
							}

							// object to save markers
							var icons = {};

							// create markers
							for (var i = 0; i < markers.markers.length; i++) {
								var markerConfig = markers.markers[i];
								var latLng = new L.LatLng(markerConfig.point.y, markerConfig.point.x);
								var marker = new L.Marker(latLng, {
									icon: markerIcon,
									clickURL: markerConfig.href,
									markerId: markerConfig.id,
									alt: markerConfig.id
								});
								var popup = new L.popup({
									offset: new L.Point(0, -markers.icon.size[1]),
									closeButton: false,
									autoPan: true
								});
								marker.addEventListener('mouseover', hoverMarker);
								marker.addEventListener('mouseout', leaveMarker);
								if (markerConfig.href) {
									marker.addEventListener('click', clickMarker);
								}
								icons[markerConfig.id] = marker;
								// bind tooltip with title content
								var label;
								if (markerConfig.img) {
									label = '<div>' +
										'<div class="marker__label p-2"> ' + markerConfig.label + '</div>' +
										'  <div class="text-center">' +
										'    <img src="' + markerConfig.img.src + '" width="' + markerConfig.img.w + '" height="' + markerConfig.img.h + '" alt="" />' +
										'  </div>' +
										'</div>';
								} else {
									label = markerConfig.label;
								}
								if (label) {
									var className = markers.tooltipClass || 'map-tooltip';
									if (markerConfig.img) {
										className += ' p-0';
									}
									marker.bindPopup(label, {
										direction: 'top',
										offset: [0, -markerIcon.options.iconSize[1]],
										opacity: 1,
										className: className,
										closeButton: false,
										autoPan: true
									});
									marker.addEventListener('mouseover', hoverMarker);
									marker.addEventListener('mouseout', leaveMarker);
								}
								markersClusterCustom.addLayer(marker);
							}
							leafmap.addLayer(markersClusterCustom);
							map.data('markers', icons);
							if (config.adjust === 'auto') {
								leafmap.fitBounds(markersClusterCustom.getBounds());
								if (markers.markers.length === 1) {
									leafmap.setZoom(config.zoom);
								}
							}
						}
					});
				});
			}
		},

		/**
		 * Single position marker management
		 */
		position: {

			init: function() {
				/* Position marker initialization */
				var map = $('.map', $(this));
				if (map.data('leafmap') === undefined) {
					map.css('height', $(window).height() - 200);
					PyAMS_GIS.init(map, {}, function (leafmap, config) {
						var data = map.data();
						var icon = L.icon({
							iconUrl: '/--static--/pyams_gis/img/marker-icon.png',
							iconSize: [25, 41],
							iconAnchor: [12, 39]
						});
						var marker = L.marker();
						marker.setIcon(icon);
						var fieldname = data.mapLeafletFieldname;
						var longitude = $('input[name="' + fieldname + '.widgets.longitude"]');
						var latitude = $('input[name="' + fieldname + '.widgets.latitude"]');
						if (longitude.val() && latitude.val()) {
							var projection = $('select[name="' + fieldname + '.widgets.projection:list"]');
							var params = {
								point: {
									longitude: parseFloat(longitude.val()),
									latitude: parseFloat(latitude.val())
								},
								from_srid: projection.val() || PyAMS_GIS.WGS_SRID,
								to_srid: PyAMS_GIS.WGS_SRID
							};
							PyAMS_GIS.callJSON('transformPoint', params, function(result) {
								if (!result.error) {
									var point = result.result.point;
									marker.setLatLng({
										lon: point.longitude,
										lat: point.latitude
									});
									marker.addTo(leafmap);
									leafmap.setView(marker.getLatLng(), config.zoom || 13);
								}
							});
						} else {
							marker.setLatLng([-90, 0]);
							marker.addTo(leafmap);
						}
						map.data('marker', marker);
						leafmap.on('click', PyAMS_GIS.position.onClick);
					});
				}
			},

			onClick: function(event) {
				var map = event.target.getContainer();
				var data = $(map).data();
				var marker = data.marker;
				var latlng = event.latlng;
				marker.setLatLng(latlng);
				var fieldname = data.mapLeafletFieldname;
				var projection = $('select[name="' + fieldname + '.widgets.projection:list"]');
				var params = {
					point: {
						longitude: latlng.lng,
						latitude: latlng.lat
					},
					from_srid: PyAMS_GIS.WGS_SRID,
					to_srid: projection.val()
				};
				PyAMS_GIS.callJSON('transformPoint', params, function(result) {
					if (!result.error) {
						var point = result.result.point;
						$('input[name="' + fieldname + '.widgets.longitude"]').val(point.longitude);
						$('input[name="' + fieldname + '.widgets.latitude"]').val(point.latitude);
						$(map).trigger('marker.changed', [map, point]);
					}
				});
			},

			changedCoordinate: function() {
				var input = $(this);
				var map = $('.map', input.parents('fieldset:first'));
				if (!map.data('marker')) {
					return;
				}
				var fieldname = map.data('map-leaflet-fieldname');
				var longitude = $('input[name="' + fieldname + '.widgets.longitude"]');
				var latitude = $('input[name="' + fieldname + '.widgets.latitude"]');
				if (longitude.val() && latitude.val()) {
					var projection = $('select[name="' + fieldname + '.widgets.projection:list"]');
					var params = {
						point: {
							longitude: parseFloat(longitude.val()),
							latitude: parseFloat(latitude.val())
						},
						from_srid: projection.val(),
						to_srid: PyAMS_GIS.WGS_SRID
					};
					PyAMS_GIS.callJSON('transformPoint', params, function (result) {
						if (!result.error) {
							var point = result.result.point;
							var marker = map.data('marker');
							marker.setLatLng({
								lon: point.longitude,
								lat: point.latitude
							});
						}
					});
				}
			},

			changedProjection: function(event) {
				var select = $(this);
				var map = $('.map', select.parents('fieldset:first'));
				var fieldname = map.data('map-leaflet-fieldname');
				var longitude = $('input[name="' + fieldname + '.widgets.longitude"]');
				var latitude = $('input[name="' + fieldname + '.widgets.latitude"]');
				if (event.removed) {
					var previous = event.removed.id;
					var current = event.added.id;
					if (previous !== current) {
						if (longitude.val() && latitude.val()) {
							var params = {
								point: {
									longitude: parseFloat(longitude.val()),
									latitude: parseFloat(latitude.val())
								},
								from_srid: previous,
								to_srid: current
							};
							PyAMS_GIS.callJSON('transformPoint', params, function (result) {
								if (!result.error) {
									var point = result.result.point;
									longitude.val(point.longitude);
									latitude.val(point.latitude);
								}
							});
						}
					}
				} else {
					PyAMS_GIS.position.changedCoordinate.apply(longitude);
				}
			},

			clear: function(event) {
				// Clear fieldset
				var fieldset = $(this).parents('fieldset:first');
				$('input', fieldset).val(null);
				// reset map position and zoom level
				var map = $('.map', fieldset);
				var marker = map.data('marker');
				if (marker) {
					marker.setLatLng([-90, 0]);
					var leafmap = map.data('leafmap');
					var config = map.data('leafmap.config');
					if (config.bounds) {
						leafmap.fitBounds(config.bounds);
					} else {
						var map_data = map.data();
						leafmap.setView(map_data.mapLeafletCenter || config.center,
										config.zoom || 13);
					}
				}
				map.trigger('marker.cleared.position', [map]);
			},

			moveMarkerTo: function(map, position, srid) {
				// Update fields
				var fieldname = map.data('map-leaflet-fieldname');
				$('input[name="' + fieldname + '.widgets.longitude"]').val(position.lon);
				$('input[name="' + fieldname + '.widgets.latitude"]').val(position.lat);
				// Get map coordinates
				var params = {
					point: {
						longitude: position.lon,
						latitude: position.lat
					},
					from_srid: srid,
					to_srid: 4326
				};
				var marker = map.data('marker');
				if (marker) {
					PyAMS_GIS.callJSON('transformPoint', params, function(result) {
						if (!result.error) {
							var point = result.result.point;
							marker.setLatLng({
								lon: point.longitude,
								lat: point.latitude
							});
							map.data('leafmap').setView(position);
						}
					});
				}
			}
		},


		/**
		 * Single rectangular area management
		 */
		area: {

			init: function () {
				var map = $('.map', $(this));
				if (map.data('leafmap') === undefined) {
					map.css('height', $(window).height() - 200);
					PyAMS_GIS.init(map, {}, function(leafmap) {
						L.Draw = L.Draw || {};
						L.Edit = L.Edit || {};
						MyAMS.ajax.check([L.Draw, L.Draw.Event, L.Map.TouchExtend, L.Edit.SimpleShape],
										[
											'/--static--/pyams_gis/js/Draw/Leaflet.draw' + MyAMS.devext + '.js',
											'/--static--/pyams_gis/js/Draw/Leaflet.Draw.Event' + MyAMS.devext + '.js',
											'/--static--/pyams_gis/js/Draw/ext/TouchEvents' + MyAMS.devext + '.js',
											'/--static--/pyams_gis/js/Draw/edit/handler/Edit.SimpleShape' + MyAMS.devext + '.js'
										],
										function() {
											MyAMS.ajax.check(L.Edit.Rectangle,
															 '/--static--/pyams_gis/js/Draw/edit/handler/Edit.Rectangle' + MyAMS.devext + '.js',
															 function () {

																function initRectangle(p1, p2) {
																	var group = new L.FeatureGroup();
																	rectangle = L.rectangle([p1, p2]);
																	group.addLayer(rectangle);
																	leafmap.addLayer(group);
																	leafmap.fitBounds(rectangle.getBounds());
																	rectangle.editing.enable();
																	map.data('area', rectangle);
																	leafmap.on(L.Draw.Event.EDITMOVE, PyAMS_GIS.area.changedArea);
																	leafmap.on(L.Draw.Event.EDITRESIZE, PyAMS_GIS.area.changedArea);
																	leafmap.on(L.Draw.Event.EDITVERTEX, PyAMS_GIS.area.changedArea);
																}

																var data = map.data();
																var fieldname = data.mapLeafletFieldname;
																var x1 = $('input[name="' + fieldname + '.widgets.x1"]');
																var y1 = $('input[name="' + fieldname + '.widgets.y1"]');
																var x2 = $('input[name="' + fieldname + '.widgets.x2"]');
																var y2 = $('input[name="' + fieldname + '.widgets.y2"]');
																var p1,
																	p2,
																	rectangle;
																if (x1.val() && y1.val() && x2.val() && y2.val()) {
																	var projection = $('select[name="' + fieldname + '.widgets.projection:list"]');
																	var params = {
																		area: {
																			x1: parseFloat(x1.val()),
																			y1: parseFloat(y1.val()),
																			x2: parseFloat(x2.val()),
																			y2: parseFloat(y2.val())
																		},
																		from_srid: projection.val(),
																		to_srid: PyAMS_GIS.WGS_SRID
																	};
																	PyAMS_GIS.callJSON('transformArea', params, function(result) {
																		if (!result.error) {
																			var area = result.result.area;
																			p1 = L.latLng({lon: area.x1, lat: area.y1});
																			p2 = L.latLng({lon: area.x2, lat: area.y2});
																			initRectangle(p1, p2);
																		}
																	});
																} else {
																	var config = map.data('leafmap.config');
																	if (config.bounds) {
																		p1 = L.latLng(config.bounds[0]);
																		p2 = L.latLng(config.bounds[1]);
																	} else {
																		p1 = L.latLng({lon: -168, lat: -56.37});
																		p2 = L.latLng({lon: 191.25, lat: 83.72});
																	}
																	initRectangle(p1, p2);
																}
															 });
										 });
					});
				}
			},

			last_event: null,

			changedArea: function(event) {
				PyAMS_GIS.area.last_event = event;
				setTimeout(function() {
					if (event === PyAMS_GIS.area.last_event) {
						var map = event.target.getContainer();
						var data = $(map).data();
						var area = data.area.getBounds();
						var fieldname = data.mapLeafletFieldname;
						var projection = $('select[name="' + fieldname + '.widgets.projection:list"]').val();
						var params = {
							area: {
								x1: area.getWest(),
								y1: area.getSouth(),
								x2: area.getEast(),
								y2: area.getNorth()
							},
							from_srid: PyAMS_GIS.WGS_SRID,
							to_srid: projection
						};
						PyAMS_GIS.callJSON('transformArea', params, function(result) {
							if (!result.error) {
								var area = result.result.area;
								$('input[name="' + fieldname + '.widgets.x1"]').val(area.x1);
								$('input[name="' + fieldname + '.widgets.y1"]').val(area.y1);
								$('input[name="' + fieldname + '.widgets.x2"]').val(area.x2);
								$('input[name="' + fieldname + '.widgets.y2"]').val(area.y2);
							}
						});
					}
				}, 100);
			},

			changedCoordinate: function() {
				var input = $(this);
				var map = $('.map', input.parents('fieldset:first'));
				if (!map.data('area')) {
					return;
				}
				var fieldname = map.data('map-leaflet-fieldname');
				var x1 = $('input[name="' + fieldname + '.widgets.x1"]');
				var y1 = $('input[name="' + fieldname + '.widgets.y1"]');
				var x2 = $('input[name="' + fieldname + '.widgets.x2"]');
				var y2 = $('input[name="' + fieldname + '.widgets.y2"]');
				if (x1.val() && y1.val() && x2.val() && y2.val()) {
					var projection = $('select[name="' + fieldname + '.widgets.projection:list"]');
					var params = {
						area: {
							x1: parseFloat(x1.val()),
							y1: parseFloat(y1.val()),
							x2: parseFloat(x2.val()),
							y2: parseFloat(y2.val())
						},
						from_srid: projection.val(),
						to_srid: 4326
					};
					PyAMS_GIS.callJSON('transformArea', params, function (result) {
						if (!result.error) {
							var area = result.result.area;
							var rect = map.data('area');
							rect.editing.disable();
							rect.setBounds([L.latLng({lon: area.x1, lat: area.y1}),
											L.latLng({lon: area.x2, lat: area.y2})]);
							rect.editing.enable();
						}
					});
				}
			},

			changedProjection: function(event) {
				var select = $(this);
				var map = $('.map', select.parents('fieldset:first'));
				var fieldname = map.data('map-leaflet-fieldname');
				var x1 = $('input[name="' + fieldname + '.widgets.x1"]');
				var y1 = $('input[name="' + fieldname + '.widgets.y1"]');
				var x2 = $('input[name="' + fieldname + '.widgets.x2"]');
				var y2 = $('input[name="' + fieldname + '.widgets.y2"]');
				if (event.removed) {
					var previous = event.removed.id;
					var current = event.added.id;
					if (previous !== current) {
						if (x1.val() && y1.val() && x2.val() && y2.val()) {
							var params = {
								area: {
									x1: parseFloat(x1.val()),
									y1: parseFloat(y1.val()),
									x2: parseFloat(x2.val()),
									y2: parseFloat(y2.val())
								},
								from_srid: previous,
								to_srid: current
							};
							PyAMS_GIS.callJSON('transformArea', params, function (result) {
								if (!result.error) {
									var area = result.result.area;
									x1.val(area.x1);
									y1.val(area.y1);
									x2.val(area.x2);
									y2.val(area.y2);
								}
							});
						}
					}
				} else {
					PyAMS_GIS.area.changedCoordinate.apply(x1);
				}
			},

			clear: function(event) {
				// Clear fieldset
				var fieldset = $(this).parents('fieldset:first');
				$('input', fieldset).val(null);
				var map = $('.map', fieldset);
				map.trigger('marker.cleared.area', [map]);
			}
		}
	};
	globals.PyAMS_GIS = PyAMS_GIS;

})(jQuery, this);
