#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
import json

from pyams_gis.interfaces.layer import IGeoJSONLayer, IMapLayer, IBaseTileMapLayer, ITileMapLayer, \
    IWMSMapLayer, IGeoportalMapLayer, \
    IEsriFeatureMapLayer, IGoogleMapLayer
from pyams_i18n.interfaces import II18n

# import packages
from persistent import Persistent
from pyams_gis import leaflet_gp, leaflet_esri, leaflet_google_mutant
from pyams_utils.dict import update_dict
from pyams_utils.fanstatic import get_resource_path
from zope.container.contained import Contained
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_gis import _


class MapLayer(Persistent, Contained):
    """Base tile map layer persistent class"""

    factory = None
    depends = {}
    layer_type = None

    name = FieldProperty(IMapLayer['name'])
    title = FieldProperty(IMapLayer['title'])
    min_zoom = FieldProperty(IMapLayer['min_zoom'])
    max_zoom = FieldProperty(IMapLayer['max_zoom'])

    def get_configuration(self):
        """Get configuration mapping"""
        result = {
            'name': self.name,
            'title': II18n(self).query_attribute('title')
        }
        update_dict(result, 'factory', self.factory)
        update_dict(result, 'minZoom', self.min_zoom)
        update_dict(result, 'maxZoom', self.max_zoom)
        if self.depends:
            depends = {}
            for name, resource in self.depends.items():
                depends[name] = get_resource_path(resource)
            update_dict(result, 'dependsOn', depends)
        return result


class GeoJSONLayer(MapLayer):
    """GeoJSON layer"""

    factory = 'PyAMS_GIS.factory.GeoJSON'
    layer_type = _("GeoJSON")

    url = FieldProperty(IGeoJSONLayer['url'])
    style = FieldProperty(IGeoJSONLayer['style'])

    def get_configuration(self):
        result = super(GeoJSONLayer, self).get_configuration()
        update_dict(result, 'url', self.url)
        if self.style:
            update_dict(result, 'style', json.loads(self.style))
        return result


class BaseTileMapLayer(MapLayer):
    """Base tile map layer"""

    attribution = FieldProperty(IBaseTileMapLayer['attribution'])
    bounds = FieldProperty(IBaseTileMapLayer['bounds'])

    def get_configuration(self):
        result = super(BaseTileMapLayer, self).get_configuration()
        update_dict(result, 'attribution', self.attribution)
        if self.bounds:
            point1, point2 = self.bounds.wgs_coordinates
            result['bounds'] = [{
                'lat': float(point1[1]),
                'lon': float(point1[0])
            }, {
                'lat': float(point2[1]),
                'lon': float(point2[0])
            }]
        return result


@implementer(ITileMapLayer)
class TileMapLayer(BaseTileMapLayer):
    """Base tile map layer persistent class"""

    factory = 'PyAMS_GIS.factory.TileLayer'
    layer_type = _("Tile")

    url = FieldProperty(ITileMapLayer['url'])

    def get_configuration(self):
        result = super(TileMapLayer, self).get_configuration()
        update_dict(result, 'url', self.url)
        return result


@implementer(IWMSMapLayer)
class WMSMapLayer(TileMapLayer):
    """WMS map mayer persistent class"""

    factory = 'PyAMS_GIS.factory.WMS'
    layer_type = _("WMS")

    crs = FieldProperty(IWMSMapLayer['crs'])
    layers = FieldProperty(IWMSMapLayer['layers'])
    styles = FieldProperty(IWMSMapLayer['styles'])
    format = FieldProperty(IWMSMapLayer['format'])
    transparent = FieldProperty(IWMSMapLayer['transparent'])
    version = FieldProperty(IWMSMapLayer['version'])
    uppercase = FieldProperty(IWMSMapLayer['uppercase'])

    def get_configuration(self):
        result = super(WMSMapLayer, self).get_configuration()
        update_dict(result, 'crs', self.crs)
        update_dict(result, 'layers', self.layers)
        update_dict(result, 'styles', self.styles)
        update_dict(result, 'format', self.format)
        update_dict(result, 'transparent', self.transparent)
        update_dict(result, 'version', self.version)
        update_dict(result, 'uppercase', self.uppercase)
        return result


@implementer(IGeoportalMapLayer)
class GeoportalMapLayer(BaseTileMapLayer):
    """Geoportal map layer persistent class"""

    factory = 'PyAMS_GIS.factory.Geoportal.WMS'
    depends = {'L.geoportalLayer.WMS': leaflet_gp}
    layer_type = _("Geoportal")

    api_key = FieldProperty(IGeoportalMapLayer['api_key'])
    layer_name = FieldProperty(IGeoportalMapLayer['layer_name'])
    crs = FieldProperty(IGeoportalMapLayer['crs'])

    def get_configuration(self):
        result = super(GeoportalMapLayer, self).get_configuration()
        update_dict(result, 'apiKey', self.api_key)
        update_dict(result, 'layer', self.layer_name)
        update_dict(result, 'crs', self.crs)
        return result


@implementer(IEsriFeatureMapLayer)
class EsriFeatureMapLayer(MapLayer):
    """ESRI feature map layer"""

    factory = 'PyAMS_GIS.factory.ESRI.Feature'
    depends = {'L.esri.featureLayer': leaflet_esri}
    layer_type = _("ESRI Features")

    url = FieldProperty(IEsriFeatureMapLayer['url'])
    token = FieldProperty(IEsriFeatureMapLayer['token'])
    where = FieldProperty(IEsriFeatureMapLayer['where'])

    def get_configuration(self):
        result = super(EsriFeatureMapLayer, self).get_configuration()
        update_dict(result, 'url', self.url)
        update_dict(result, 'token', self.token)
        update_dict(result, 'where', self.where)
        return result


@implementer(IGoogleMapLayer)
class GoogleMapLayer(MapLayer):
    """Google maps layer"""

    factory = 'PyAMS_GIS.factory.Google'
    depends = {'L.gridLayer.googleMutant': leaflet_google_mutant}
    layer_type = _("Google")

    api_key = FieldProperty(IGoogleMapLayer['api_key'])
    type = FieldProperty(IGoogleMapLayer['type'])

    def get_configuration(self):
        result = super(GoogleMapLayer, self).get_configuration()
        update_dict(result, 'apiKey', self.api_key)
        update_dict(result, 'type', self.type)
        return result
