#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.url import resource_url
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.component.interfaces import ISite
from zope.interface import Interface, implementer

from pyams_form.form import AJAXEditForm
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_gis.zmi.interfaces import IMapManagerMenu
from pyams_i18n.column import I18nAttrColumn
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import ContainerView, delete_container_element
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import BaseTable, DefaultElementEditorAdapter, I18nColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import MANAGE_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import IControlPanelMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'


from pyams_gis import _


@adapter_config(context=(IMapManager, IAdminLayer, Interface), provides=ITableElementEditor)
class MapManagerTableElementEditor(DefaultElementEditorAdapter):
    """Map manager table element editor"""

    view_name = 'map-manager.html'
    modal_target = False

    @property
    def url(self):
        site = get_parent(self.context, ISite)
        return resource_url(site, self.request, 'admin#{0}'.format(self.view_name))


@viewlet_config(name='map-manager.menu', context=ISite, layer=IAdminLayer, manager=IControlPanelMenu,
                permission=MANAGE_SYSTEM_PERMISSION, weight=6)
@viewletmanager_config(name='map-manager.menu', context=ISite, layer=IAdminLayer)
@implementer(IMapManagerMenu)
class MapManagerMenuItem(MenuItem):
    """Map manager menu"""

    label = _("Maps manager")
    icon_class = 'fa-map-marker'
    url = '#map-manager.html'


class MapManagerLayersTable(BaseTable):
    """Map manager layers table"""

    id = 'map_manager_layers_table'
    title = _("Map manager layers")

    @reify
    def data_attributes(self):
        manager = query_utility(IMapManager)
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(manager, self.request)
        })
        return attributes


@adapter_config(name='name', context=(Interface, IAdminLayer, MapManagerLayersTable), provides=IColumn)
class MapManagerLayersNameColumn(I18nColumn, GetAttrColumn):
    """Map manager layers name column"""

    _header = _("Name")
    attrName = 'name'
    weight = 10


@adapter_config(name='layer_type', context=(Interface, IAdminLayer, MapManagerLayersTable), provides=IColumn)
class MapManagerLayersTypeColumn(I18nColumn, GetAttrColumn):
    """Map manager layers type column"""

    _header = _("Type")
    attrName = 'layer_type'
    weight = 15
    
    def getValue(self, obj):
        return self.request.localizer.translate(super(MapManagerLayersTypeColumn, self).getValue(obj))


@adapter_config(name='title', context=(Interface, IAdminLayer, MapManagerLayersTable), provides=IColumn)
class MapManagerLayersTitleColumn(I18nColumn, I18nAttrColumn):
    """Map manager layers title column"""

    _header = _("Title")
    attrName = 'title'
    defaultValue = '--'
    weight = 20


@adapter_config(name='trash', context=(Interface, IAdminLayer, MapManagerLayersTable), provides=IColumn)
class MapManagerLayersTrashCOlumn(TrashColumn):
    """Map manager layer trash column"""

    icon_hint = _("Delete layer")
    permission = MANAGE_SYSTEM_PERMISSION


@view_config(name='delete-element.json', context=IMapManager, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
def delete_map_manager_layer(request):
    """Delete layer from map manager"""
    return delete_container_element(request, ignore_permission=True)


@adapter_config(context=(ISite, IAdminLayer, MapManagerLayersTable), provides=IValues)
class MapManagerValuesAdapter(ContextRequestViewAdapter):
    """Map manager layers table values adapter"""

    @property
    def values(self):
        manager = query_utility(IMapManager)
        if manager is not None:
            return list(manager.values())
        return ()


@pagelet_config(name='map-manager.html', context=ISite, layer=IPyAMSLayer, permission=MANAGE_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class MapManagerView(AdminView, ContainerView):
    """Map manager layers view"""

    table_class = MapManagerLayersTable


@adapter_config(context=(ISite, IAdminLayer, MapManagerView), provides=IPageHeader)
class MapManagerHeaderAdapter(DefaultPageHeaderAdapter):
    """Map manager view header adapter"""

    icon_class = 'fa fa-fw fa-map-marker'
    subtitle = _("Maps manager")


@viewlet_config(name='map-manager.configuration.menu', context=ISite, layer=IAdminLayer,
                manager=IMapManagerMenu, permission=MANAGE_SYSTEM_PERMISSION, weight=1)
class MapManagerConfigurationMenuItem(MenuItem):
    """Map manager properties menu"""

    label = _("Configuration...")
    url = 'configuration.html'
    modal_target = True

    def get_url(self):
        manager = query_utility(IMapManager)
        return resource_url(manager, self.request, self.url)


@pagelet_config(name='configuration.html', context=IMapManager, layer=IPyAMSLayer, permission=MANAGE_SYSTEM_PERMISSION)
class MapManagerConfigurationEditForm(AdminDialogEditForm):
    """Map manager edit form"""

    prefix = 'map_configuration.'

    title = _("Maps manager")
    legend = _("Default map configuration")
    icon_css_class = 'fa fa-fw fa-map-marker'
    dialog_class = 'modal-large'

    fields = field.Fields(IMapConfiguration)

    ajax_handler = 'configuration.json'
    edit_permission = MANAGE_SYSTEM_PERMISSION

    def getContent(self):
        return query_utility(IMapManager)


@view_config(name='configuration.json', context=IMapManager, request_type=IPyAMSLayer,
             permission=MANAGE_SYSTEM_PERMISSION, renderer='json', xhr=True)
class MapManagerConfigurationAJAXEditForm(AJAXEditForm, MapManagerConfigurationEditForm):
    """Map manager edit form, JSON renderer"""


@view_config(name='get-map-configuration.json', context=Interface, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def get_map_configuration(request):
    """Get map configuration in JSON format"""
    manager = query_utility(IMapManager)
    if manager is not None:
        configuration = IMapConfiguration(manager)
        if configuration is not None:
            return configuration.get_configuration()
