#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from decimal import Decimal

try:
    from osgeo.osr import SpatialReference, CoordinateTransformation
    have_gdal = True
except ImportError:
    have_gdal = False
from persistent import Persistent

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_gis.interfaces import IGeoPoint, IGeoPointZ, WGS84

__docformat__ = 'restructuredtext'


@implementer(IGeoPoint)
class GeoPoint(Persistent):
    """GeoPoint attribute object"""

    longitude = FieldProperty(IGeoPoint['longitude'])
    latitude = FieldProperty(IGeoPoint['latitude'])
    projection = FieldProperty(IGeoPoint['projection'])

    def __init__(self, data=None, **kwargs):
        super().__init__()
        if 'longitude' in kwargs:
            self.longitude = Decimal(kwargs.get('longitude'))
        if 'latitude' in kwargs:
            self.latitude = Decimal(kwargs.get('latitude'))
        if 'projection' in kwargs:
            self.projection = Decimal(kwargs.get('projection'))

    def __bool__(self):
        return bool(self.longitude and self.latitude)

    def get_coordinates(self, projection=WGS84):
        if projection == self.projection:
            return self.longitude, self.latitude
        if (not have_gdal) or not self:
            return None, None
        source = SpatialReference()
        source.ImportFromEPSG(self.projection)
        destination = SpatialReference()
        destination.ImportFromEPSG(projection)
        transformation = CoordinateTransformation(source, destination)
        return transformation.TransformPoint(float(self.longitude), float(self.latitude))[0:2]

    @property
    def wgs_coordinates(self):
        return self.get_coordinates(WGS84)

    def to_json(self):
        if not self:
            return None
        return {
            'x': float(self.longitude),
            'y': float(self.latitude),
            'crs': float(self.projection)
        }


@implementer(IGeoPointZ)
class GeoPointZ(GeoPoint):
    """GeoPointZ attribute object"""

    altitude = FieldProperty(IGeoPointZ['altitude'])

    def __init__(self, data=None, **kwargs):
        super().__init__(data, **kwargs)
        if 'altitude' in kwargs:
            self.altitude = kwargs.get('altitude')

    def to_json(self):
        result = super().to_json()
        if result:
            result.update({
                'z': float(self.altitude)
            })
        return result
