#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

try:
    from osgeo.osr import SpatialReference, CoordinateTransformation
    have_gdal = True
except ImportError:
    have_gdal = False
from persistent import Persistent

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_gis.interfaces import WGS84, IGeoArea

__docformat__ = 'restructuredtext'


@implementer(IGeoArea)
class GeoArea(Persistent):
    """GeoArea attribute object"""

    x1 = FieldProperty(IGeoArea['x1'])
    y1 = FieldProperty(IGeoArea['y1'])
    x2 = FieldProperty(IGeoArea['x2'])
    y2 = FieldProperty(IGeoArea['y2'])
    projection = FieldProperty(IGeoArea['projection'])

    def __init__(self, data=None, **kwargs):
        super().__init__()
        if 'x1' in kwargs:
            self.x1 = kwargs['x1']
        if 'y1' in kwargs:
            self.y1 = kwargs['y1']
        if 'x2' in kwargs:
            self.x2 = kwargs['x2']
        if 'y2' in kwargs:
            self.y2 = kwargs['y2']

    def __bool__(self):
        return bool(self.x1 and self.y1 and self.x2 and self.y2)

    def get_coordinates(self, projection=WGS84):
        if projection == self.projection:
            return (self.x1, self.y1), (self.x2, self.y2)
        if (not have_gdal) or not self:
            return None, None
        source = SpatialReference()
        source.ImportFromEPSG(self.projection)
        destination = SpatialReference()
        destination.ImportFromEPSG(projection)
        transformation = CoordinateTransformation(source, destination)
        return transformation.TransformPoint(float(self.x1), float(self.y1))[0:2], \
               transformation.TransformPoint(float(self.x2), float(self.y2))[0:2]

    @property
    def wgs_coordinates(self):
        return self.get_coordinates(WGS84)

    def to_json(self):
        if not self:
            return None
        return {
            'x1': float(self.x1),
            'y1': float(self.y1),
            'x2': float(self.x2),
            'y2': float(self.y2),
            'projection': self.projection
        }
