#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from collections import OrderedDict

import transaction
from pyramid.view import view_config
from transaction.interfaces import ITransactionManager
from z3c.form import button, field
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, implementer
from zope.schema import Bool, Int
from zope.schema.fieldproperty import FieldProperty

from pyams_file.image import ThumbnailGeometry
from pyams_file.interfaces import IFileInfo, IFileModifierForm, IImage, IMediaWidget, \
    IResponsiveImage, IThumbnailForm, IThumbnailGeometry, IThumbnailer, IThumbnails
from pyams_file.zmi import FileModifierAction
from pyams_form.form import ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IWidgetsPrefixViewletsManager
from pyams_form.schema import CloseButton
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_image_refresh_event
from pyams_skin.interfaces.viewlet import IContextActions
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarActionItem
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import ICacheKeyValue, VIEW_PERMISSION
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm, AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_file import _


#
# Image rotate
#

@viewlet_config(name='image-rotate.action', context=IImage, layer=IPyAMSLayer, view=IMediaWidget,
                manager=IContextActions, weight=20)
class ImageRotateAction(FileModifierAction):
    """Image rotate action"""

    label = _("Rotate image to right...")
    label_css_class = 'fa fa-fw-md fa-rotate-right'

    @property
    def url(self):
        return 'MyAMS.ajax.getJSON?url={0}'.format(
            absolute_url(self.context, self.request, 'rotate.json'))

    def get_url(self):
        return self.url


@view_config(name='rotate.json', context=IImage, request_type=IPyAMSLayer,
             permission=VIEW_PERMISSION, renderer='json', xhr=True)
def rotate_image(request):
    """Rotate given image to right"""
    image = request.context
    IImage(image).rotate(-90)
    # Commit to save blobs!!
    ITransactionManager(image).commit()
    cache_key = ICacheKeyValue(image)
    thumbnail = IThumbnails(image).get_thumbnail('128x128')
    dc = IZopeDublinCore(thumbnail)
    return {
        'status': 'success',
        'message': request.localizer.translate(AdminDialogEditForm.successMessage),
        'callbacks': [{
            'callback': 'MyAMS.skin.refreshContent',
            'options': {
                'object_id': 'thumbnail_{0}'.format(cache_key),
                'content': '<img class="thumbnail" id="thumbnail_{key}" '
                           'src="{src}" title="{title}" />'.format(
                    key=cache_key,
                    src='{0}?_={1}'.format(absolute_url(thumbnail, request),
                                           dc.modified.timestamp()),
                    title=II18n(image).query_attribute('title', request=request)
                )
            }
        }],
        'close_form': False
    }


#
# Image crop
#

@viewlet_config(name='image.crop.action', context=IImage, layer=IPyAMSLayer, view=IMediaWidget,
                manager=IContextActions, weight=21)
class ImageCropAction(FileModifierAction):
    """Image crop action"""

    label = _("Crop image...")
    label_css_class = 'fa fa-fw-md fa-crop'

    url = 'crop.html'
    modal_target = True


class ICropFormButtons(Interface):
    """Image crop form buttons"""

    close = CloseButton(name='close', title=_("Close"))
    crop = button.Button(name='crop', title=_("Crop"))


@pagelet_config(name='crop.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='crop.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IFileModifierForm)
class ImageCropForm(AdminDialogEditForm):
    """Image crop form"""

    prefix = 'image_crop.'

    legend = _("Crop image")
    icon_css_class = 'fa fa-fw fa-crop'
    dialog_class = 'modal-large'

    fields = field.Fields(Interface)
    buttons = button.Buttons(ICropFormButtons)
    edit_permission = None

    @property
    def title(self):
        return self.context.title or self.context.filename

    @property
    def timestamp(self):
        dc = IZopeDublinCore(self.context)
        return dc.modified.timestamp()

    def updateActions(self):
        super(ImageCropForm, self).updateActions()
        if 'crop' in self.actions:
            self.actions['crop'].addClass('btn-primary')

    def update_content(self, content, data):
        image_size = self.context.get_image_size()
        x1 = int(self.request.params.get('selection.x1', 0))
        y1 = int(self.request.params.get('selection.y1', 0))
        x2 = int(self.request.params.get('selection.x2', image_size[0]))
        y2 = int(self.request.params.get('selection.y2', image_size[1]))
        self.context.crop(x1, y1, x2, y2)

    def get_ajax_output(self, changes):
        ITransactionManager(self.context).commit()
        thumbnail = IThumbnails(self.context).get_thumbnail('128x128')
        return {
            'status': 'success',
            'message': self.request.localizer.translate(self.successMessage),
            'events': [
                get_json_image_refresh_event(thumbnail, self.request, self.context)
            ]
        }


@viewlet_config(name='crop.widgets-prefix', context=IImage, layer=IAdminLayer, view=ImageCropForm,
                manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/image-crop.pt')
class ImageCropViewletsPrefix(Viewlet):
    """Image crop viewlets prefix"""


@adapter_config(context=(IImage, IAdminLayer, ImageCropForm), provides=IFormHelp)
class ImageCropFormHelpAdapter(FormHelp):
    """Image crop form help adapter"""

    message = _("""You can use this form to crop an image.

**WARNING**: cropping an image will reset all selected thumbnails and adaptive images!!""")
    message_format = 'rest'


#
# Generic thumbnails selection forms components
#

@adapter_config(context=(IImage, IAdminLayer, IThumbnailForm), provides=IFormHelp)
class ThumbnailFormHelpAdapter(FormHelp):
    """Thumbnail form help adapter"""

    message = _("""You can use this form to make a selection on an image.""")
    message_format = 'rest'


class IThumbnailFormButtons(Interface):
    """Image crop form buttons"""

    close = CloseButton(name='close', title=_("Close"))
    crop = button.Button(name='crop', title=_("Select thumbnail"))


class ImageSelectionThumbnailEditForm(AdminDialogEditForm):
    """Image portrait thumbnail edit form"""

    dialog_class = 'modal-large'

    fields = field.Fields(Interface)
    buttons = button.Buttons(IThumbnailFormButtons)
    edit_permission = None

    selection_name = None
    selection_ratio = None

    @property
    def title(self):
        return self.context.title or self.context.filename

    def updateActions(self):
        super(ImageSelectionThumbnailEditForm, self).updateActions()
        if 'crop' in self.actions:
            self.actions['crop'].addClass('btn-primary')

    def update_content(self, content, data):
        geometry = ThumbnailGeometry()
        geometry.x1 = int(self.request.params.get('selection.x1'))
        geometry.y1 = int(self.request.params.get('selection.y1'))
        geometry.x2 = int(self.request.params.get('selection.x2'))
        geometry.y2 = int(self.request.params.get('selection.y2'))
        if not geometry.is_empty():
            IThumbnails(self.context).set_geometry(self.selection_name, geometry)
            return {IThumbnailGeometry: ('x1', 'y1', 'x2', 'y2')}

    def get_ajax_output(self, changes):
        if changes:
            return {
                'status': 'success',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return {
                'status': 'info',
                'message': self.request.localizer.translate(self.noChangesMessage)
            }


@viewlet_config(name='thumbnail.selection.widgets-prefix', context=IImage, layer=IAdminLayer,
                view=ImageSelectionThumbnailEditForm, manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/image-selection.pt')
class ImageSelectionThumbnailViewletsPrefix(Viewlet):
    """Image square thumbnail viewlets prefix"""

    @property
    def timestamp(self):
        dc = IZopeDublinCore(self.context)
        return dc.modified.timestamp()


#
# Image portrait thumbnail selection
#

@viewlet_config(name='image.thumb.portrait.action', context=IImage, layer=IAdminLayer, view=IMediaWidget,
                manager=IContextActions, weight=55)
class ImagePortraitThumbnailAction(FileModifierAction):
    """Portrait thumbnail image selection"""

    _label = _("Select portrait thumbnail...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++portrait:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-address-book-o'

    url = 'portrait-thumbnail.html'
    modal_target = True


@pagelet_config(name='portrait-thumbnail.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='portrait-thumbnail.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IThumbnailForm)
class ImagePortraitThumbnailEditForm(ImageSelectionThumbnailEditForm):
    """Image portrait thumbnail edit form"""

    prefix = 'image_portrait.'

    legend = _("Select portrait thumbnail")
    icon_css_class = 'fa fa-fw fa-address-book-o'

    selection_name = 'portrait'
    selection_ratio = '3:4'


@adapter_config(context=(IImage, IAdminLayer, ImagePortraitThumbnailEditForm), provides=IFormHelp)
class ImagePortraitThumbnailEditFormHelpAdapter(FormHelp):
    """Portraitramic image thumbnail edit form help adapter"""

    message = _("""You can use this form to select a portrait thumbnail of this image.

**WARNING**: cropping or resizing an image will reset all selected thumbnails and adaptive images!!""")
    message_format = 'rest'


#
# Image square thumbnail selection
#

@viewlet_config(name='image.thumb.square.action', context=IImage, layer=IPyAMSLayer, view=IMediaWidget,
                manager=IContextActions, weight=60)
class ImageSquareThumbnailAction(FileModifierAction):
    """Square thumbnail image selection"""

    _label = _("Select square thumbnail...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++square:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-instagram'

    url = 'square-thumbnail.html'
    modal_target = True


@pagelet_config(name='square-thumbnail.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='square-thumbnail.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IThumbnailForm)
class ImageSquareThumbnailEditForm(ImageSelectionThumbnailEditForm):
    """Image square thumbnail edit form"""

    prefix = 'image_square.'

    legend = _("Select square thumbnail")
    icon_css_class = 'fa fa-fw fa-instagram'

    selection_name = 'square'
    selection_ratio = '1:1'


@adapter_config(context=(IImage, IAdminLayer, ImageSquareThumbnailEditForm), provides=IFormHelp)
class ImageSquareThumbnailEditFormHelpAdapter(FormHelp):
    """Square image thumbnail edit form help adapter"""

    message = _("""You can use this form to select a square thumbnail of this image.

**WARNING**: cropping or resizing an image will reset all selected thumbnails and adaptive images!!""")
    message_format = 'rest'


#
# Image panoramic thumbnail selection
#

@viewlet_config(name='image.thumb.pano.action', context=IImage, layer=IAdminLayer, view=IMediaWidget,
                manager=IContextActions, weight=65)
class ImagePanoThumbnailAction(FileModifierAction):
    """Panoramic thumbnail image selection"""

    _label = _("Select panoramic thumbnail...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++pano:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-youtube-play'

    url = 'pano-thumbnail.html'
    modal_target = True


@pagelet_config(name='pano-thumbnail.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='pano-thumbnail.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IThumbnailForm)
class ImagePanoThumbnailEditForm(ImageSelectionThumbnailEditForm):
    """Image panoramic thumbnail edit form"""

    prefix = 'image_pano.'

    legend = _("Select panoramic thumbnail")
    icon_css_class = 'fa fa-fw fa-youtube-play'

    selection_name = 'pano'
    selection_ratio = '16:9'


@adapter_config(context=(IImage, IAdminLayer, ImagePanoThumbnailEditForm), provides=IFormHelp)
class ImagePanoThumbnailEditFormHelpAdapter(FormHelp):
    """Panoramic image thumbnail edit form help adapter"""

    message = _("""You can use this form to select a panoramic thumbnail of this image.

**WARNING**: cropping or resizing an image will reset all selected thumbnails and adaptive images!!""")
    message_format = 'rest'


#
# Image banner thumbnail selection
#

@viewlet_config(name='image.thumb.banner.action', context=IImage, layer=IAdminLayer, view=IMediaWidget,
                manager=IContextActions, weight=70)
class ImageBannerThumbnailAction(FileModifierAction):
    """Banner thumbnail image selection"""

    _label = _("Select banner thumbnail...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++banner:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-window-minimize fa-rotate-180'

    url = 'banner-thumbnail.html'
    modal_target = True


@pagelet_config(name='banner-thumbnail.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='banner-thumbnail.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IThumbnailForm)
class ImageBannerThumbnailEditForm(ImageSelectionThumbnailEditForm):
    """Image banner thumbnail edit form"""

    prefix = 'image_banner.'

    legend = _("Select banner thumbnail")
    icon_css_class = 'fa fa-fw fa-window-minimize fa-rotate-180'

    selection_name = 'banner'
    selection_ratio = '5:1'


@adapter_config(context=(IImage, IAdminLayer, ImageBannerThumbnailEditForm), provides=IFormHelp)
class ImageBannerThumbnailEditFormHelpAdapter(FormHelp):
    """Banner image thumbnail edit form help adapter"""

    message = _("""You can use this form to select a banner thumbnail of this image.

**WARNING**: cropping or resizing an image will reset all selected thumbnails and adaptive images!!""")
    message_format = 'rest'


#
# Extra-small devices selection
#

@viewlet_config(name='responsive-image.selection.xs.action', context=IResponsiveImage, layer=IAdminLayer,
                view=IMediaWidget, manager=IContextActions, weight=41)
class ResponsiveImageXsSelectionAction(FileModifierAction):
    """Responsive image XS selection"""

    _label = _("Select responsive XS image...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++xs:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-mobile'

    url = 'selection-xs.html'
    modal_target = True


@pagelet_config(name='selection-xs.html', context=IResponsiveImage, layer=IPyAMSLayer,
                permission=VIEW_PERMISSION)
@ajax_config(name='selection-xs.json', context=IResponsiveImage, layer=IPyAMSLayer,
             permission=VIEW_PERMISSION)
class ResponsiveImageXsSelectionForm(ImageSelectionThumbnailEditForm):
    """Responsive image XS selection edit form"""

    legend = _("Select image for extra-small (XS) devices")
    icon_css_class = 'fa fa-fw fa-mobile'

    selection_name = 'xs'


#
# Small devices selection
#

@viewlet_config(name='responsive-image.selection.sm.action', context=IResponsiveImage, layer=IAdminLayer,
                view=IMediaWidget, manager=IContextActions, weight=42)
class ResponsiveImageSmSelectionAction(FileModifierAction):
    """Responsive image SM selection"""

    _label = _("Select responsive SM image...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++sm:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-tablet'

    url = 'selection-sm.html'
    modal_target = True


@pagelet_config(name='selection-sm.html', context=IResponsiveImage, layer=IPyAMSLayer,
                permission=VIEW_PERMISSION)
@ajax_config(name='selection-sm.json', context=IResponsiveImage, layer=IPyAMSLayer,
             permission=VIEW_PERMISSION)
class ResponsiveImageSmSelectionForm(ImageSelectionThumbnailEditForm):
    """Responsive image SM selection edit form"""

    legend = _("Select image for small (SM) devices")
    icon_css_class = 'fa fa-fw fa-tablet'

    selection_name = 'sm'


#
# Medium devices selection
#

@viewlet_config(name='responsive-image.selection.md.action', context=IResponsiveImage, layer=IAdminLayer,
                view=IMediaWidget, manager=IContextActions, weight=43)
class ResponsiveImageMdSelectionAction(FileModifierAction):
    """Responsive image MD selection"""

    _label = _("Select responsive MD image...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++md:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-desktop'

    url = 'selection-md.html'
    modal_target = True


@pagelet_config(name='selection-md.html', context=IResponsiveImage, layer=IPyAMSLayer,
                permission=VIEW_PERMISSION)
@ajax_config(name='selection-md.json', context=IResponsiveImage, layer=IPyAMSLayer,
             permission=VIEW_PERMISSION)
class ResponsiveImageMdSelectionForm(ImageSelectionThumbnailEditForm):
    """Responsive image MD selection edit form"""

    legend = _("Select image for medium (MD) devices")
    icon_css_class = 'fa fa-fw fa-desktop'

    selection_name = 'md'


#
# Large devices selection
#

@viewlet_config(name='responsive-image.selection.lg.action', context=IResponsiveImage, layer=IAdminLayer,
                view=IMediaWidget, manager=IContextActions, weight=44)
class ResponsiveImageLgSelectionAction(FileModifierAction):
    """Responsive image LG selection"""

    _label = _("Select responsive LG image...")

    @property
    def label(self):
        return '<div>{label}</div><div class="padding-y-5"><img src="{url}/++thumb++lg:200x128?_=" ' \
               '/></div>'.format(label=self.request.localizer.translate(self._label),
                                 url=absolute_url(self.context, self.request))

    label_css_class = 'fa fa-fw-md fa-television'

    url = 'selection-lg.html'
    modal_target = True


@pagelet_config(name='selection-lg.html', context=IResponsiveImage, layer=IPyAMSLayer,
                permission=VIEW_PERMISSION)
@ajax_config(name='selection-lg.json', context=IResponsiveImage, layer=IPyAMSLayer,
             permission=VIEW_PERMISSION)
class ResponsiveImageLgSelectionForm(ImageSelectionThumbnailEditForm):
    """Responsive image LG selection edit form"""

    legend = _("Select image for large (LG) devices")
    icon_css_class = 'fa fa-fw fa-television'

    selection_name = 'lg'


#
# See all thumbnails!!
#

@viewlet_config(name='display-all-thumbnails.action', context=IImage, layer=IAdminLayer,
                view=IMediaWidget, manager=IContextActions, weight=80)
class ImageThumbnailsDisplayAction(ToolbarActionItem):
    """Image thumbnails display action"""

    label = _("Display all thumbnails")
    label_css_class = 'fa fa-fw fa-th-large text-primary'
    hint_gravity = 'nw'

    url = 'display-all-thumbnails.html'
    modal_target = True


@pagelet_config(name='display-all-thumbnails.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
class ImageThumbnailsDisplayForm(AdminDialogDisplayForm):
    """Image thumbnails display form"""

    @property
    def title(self):
        info = IFileInfo(self.context)
        return info.title or info.filename

    legend = _("Display all image thumbnails")
    icon_css_class = 'fa fa-fw fa-th-large'
    dialog_class = 'modal-large'

    fields = field.Fields(Interface)


@viewlet_config(name='image-thumbnails', context=IImage, layer=IPyAMSLayer,
                view=ImageThumbnailsDisplayForm, manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/image-thumbnails.pt')
class ImageThumbnailsViewletsPrefix(Viewlet):
    """Image thumbnails viewlets prefix"""

    def get_thumbnails(self):
        registry = self.request.registry
        translate = self.request.localizer.translate
        thumbnailers = OrderedDict()
        adapters = sorted(registry.getAdapters((self.context,), IThumbnailer),
                          key=lambda x: x[1].weight)
        for name, adapter in adapters:
            thumbnails = IThumbnails(self.context)
            if name:
                selection = thumbnails.get_selection(name)
                transaction.commit()
                thumbnails = IThumbnails(selection)
            thumbnail = thumbnails.get_thumbnail('200x128')
            thumbnailers.setdefault(translate(adapter.section), []).append({'name': name,
                                                                            'label': translate(adapter.label),
                                                                            'thumbnail': thumbnail})
        transaction.commit()
        return thumbnailers


#
# Image resize
#

@viewlet_config(name='image.resize.action', context=IImage, layer=IPyAMSLayer, view=IMediaWidget,
                manager=IContextActions, weight=90)
class ImageResizeAction(FileModifierAction):
    """Image resize action"""

    label = _("Resize image...")
    label_css_class = 'fa fa-fw-md fa-compress'

    url = 'resize.html'
    modal_target = True


class IResizeFormButtons(Interface):
    """Image resize form buttons"""

    close = CloseButton(name='close', title=_("Close"))
    resize = button.Button(name='resize', title=_("Resize image"))


class IImageResizeInfo(Interface):
    """Image resize interface"""

    width = Int(title=_("New image width"))

    height = Int(title=_("New image height"))

    keep_ratio = Bool(title=_("Keep aspect ratio"),
                      description=_("Check to keep original aspect ratio; image will be resized as large as "
                                    "possible within given limits"),
                      required=True,
                      default=True)


@implementer(IImageResizeInfo)
class ImageResizeInfo(object):
    """Image resize info"""

    width = FieldProperty(IImageResizeInfo['width'])
    height = FieldProperty(IImageResizeInfo['height'])
    keep_ratio = FieldProperty(IImageResizeInfo['keep_ratio'])


@adapter_config(context=IImage, provides=IImageResizeInfo)
def image_resize_info_factory(image):
    size = image.get_image_size()
    info = ImageResizeInfo()
    info.width = size[0]
    info.height = size[1]
    return info


@pagelet_config(name='resize.html', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@ajax_config(name='resize.json', context=IImage, layer=IPyAMSLayer, permission=VIEW_PERMISSION)
@implementer(IFileModifierForm)
class ImageResizeForm(AdminDialogEditForm):
    """Image resize form"""

    prefix = 'image_resize.'

    legend = _("Resize image")
    icon_css_class = 'fa fa-fw fa-compress'

    fields = field.Fields(IImageResizeInfo)
    buttons = button.Buttons(IResizeFormButtons)
    edit_permission = None

    @property
    def title(self):
        return self.context.title or self.context.filename

    def updateActions(self):
        super(ImageResizeForm, self).updateActions()
        if 'resize' in self.actions:
            self.actions['resize'].addClass('btn-primary')

    def update_content(self, content, data):
        data = data.get(self, data)
        self.context.resize(data.get('width'), data.get('height'), data.get('keep_ratio'))

    def get_ajax_output(self, changes):
        return {
            'status': 'reload',
            'message': self.request.localizer.translate(self.successMessage)
        }


@adapter_config(context=(IImage, IAdminLayer, ImageResizeForm), provides=IFormHelp)
class ImageResizeFormHelpAdapter(FormHelp):
    """Image resize form help adapter"""

    message = _("""You can use this form to change image dimensions.

This will generate a new image only if requested size is smaller than the original one.""")
    message_format = 'rest'
