#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from pyams_utils.interfaces.data import IObjectData

__docformat__ = 'restructuredtext'

from z3c.form import field
from z3c.form.browser.textarea import TextAreaFieldWidget
from zope.interface import Interface, implementer, alsoProvides

from pyams_file.interfaces import IFile, IFileInfo, IFileModifierForm
from pyams_file.zmi import FileModifierAction
from pyams_form.form import ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IContextActions
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION, MANAGE_SYSTEM_PERMISSION
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm

from pyams_file import _


#
# File properties form
#

@viewlet_config(name='file.properties.action', context=IFile, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, permission=MANAGE_SYSTEM_PERMISSION, weight=1)
class FilePropertiesAction(FileModifierAction):
    """File properties action"""

    label = _("Properties...")
    label_css_class = 'fa fa-fw-md fa-edit'

    url = 'properties.html'
    modal_target = True


@pagelet_config(name='properties.html', context=IFile, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IFile, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IFileModifierForm)
class FilePropertiesEditForm(AdminDialogEditForm):
    """File properties edit form"""

    prefix = 'file_properties.'

    legend = _("Update file properties")
    icon_css_class = 'fa fa-fw fa-edit'

    fields = field.Fields(IFileInfo)
    edit_permission = None

    @property
    def title(self):
        return self.context.title or self.context.filename

    def get_ajax_output(self, changes):
        info_changes = changes.get(IFileInfo, ())
        if ('title' in info_changes) or ('filename' in info_changes):
            return {
                'status': 'reload',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return super(self.__class__, self).get_ajax_output(changes)


#
# Text file editor
#

@viewlet_config(name='file.editor.action', context=IFile, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, weight=2)
class FileContentEditorAction(FileModifierAction):
    """Text file content editor action"""

    def __new__(cls, context, request, view, manager):
        if not (context.content_type.startswith('text/') or context.content_type.startswith('image/svg')):
            return None
        return FileModifierAction.__new__(cls)

    label = _("Edit file...")
    label_css_class = 'fa fa-fw-md fa-pencil'

    url = 'edit-content.html'
    modal_target = True


@pagelet_config(name='edit-content.html', context=IFile, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='edit-content.json', context=IFile, layer=IPyAMSLayer)
@implementer(IFileModifierForm)
class FileContentEditorForm(AdminDialogEditForm):
    """Text file content edit form"""

    prefix = 'file_content.'
    dialog_class = 'modal-max'

    legend = _("Edit file content")
    icon_css_class = 'fa fa-fw fa-pencil'
    label_css_class = 'control-label col-md-1'
    input_css_class = 'col-md-11'

    fields = field.Fields(IFile).select('data')
    fields['data'].widgetFactory = TextAreaFieldWidget

    edit_permission = None

    def updateWidgets(self, prefix=None):
        super(FileContentEditorForm, self).updateWidgets(prefix)
        if 'data' in self.widgets:
            widget = self.widgets['data']
            widget.widget_css_class = 'input height-500 monospace text-editor'
            widget.object_data = {
                'ams-editor-filename': self.context.filename
            }
            alsoProvides(widget, IObjectData)
