#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import os
from cgi import FieldStorage
from datetime import datetime

from pyramid.interfaces import IView
from z3c.form.browser.file import FileWidget as BaseFileWidget
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import IDataConverter, IFieldWidget, NOT_CHANGED
from z3c.form.util import toBytes
from z3c.form.widget import FieldWidget
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import implementer_only

from pyams_file.file import EXTENSIONS_THUMBNAILS
from pyams_file.interfaces import DELETED_FILE, IFileField, IFileWidget, IMediaField, \
    IMediaWidget, IThumbnailMediaField, IThumbnailMediaWidget, IThumbnails
from pyams_form.interfaces.form import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config
from pyams_utils.size import get_human_size
from pyams_utils.url import absolute_url


__docformat__ = 'restructuredtext'


@adapter_config(context=(IFileField, IFileWidget), provides=IDataConverter)
class FileUploadDataConverter(BaseDataConverter):
    """File upload data converter"""

    def toWidgetValue(self, value):
        return value

    def toFieldValue(self, value):
        deleted_field_name = '{0}__deleted'.format(self.widget.name)
        deleted = self.widget.request.params.get(deleted_field_name)
        if deleted:
            return DELETED_FILE
        if (value is None) or (value is NOT_CHANGED) or (value == ''):
            return NOT_CHANGED
        elif isinstance(value, FieldStorage):
            return value.filename, value.file
        elif isinstance(value, tuple):
            return value
        else:
            return toBytes(value)


#
# File widgets
#

@widgettemplate_config(mode='input', template='templates/file-input.pt', widget=IFileWidget, layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/file-display.pt', widget=IFileWidget, layer=IFormLayer)
@implementer_only(IFileWidget)
class FileWidget(BaseFileWidget):
    """File widget"""

    @property
    def timestamp(self):
        dc = IZopeDublinCore(self.current_value, None)
        if dc is None:
            return datetime.utcnow().timestamp()
        else:
            return dc.modified.timestamp()

    @property
    def current_value(self):
        if self.form.ignoreContext:
            return None
        lang = getattr(self, 'lang', None)
        if lang is None:
            return self.field.get(self.context)
        else:
            value = self.field.get(self.context)
            if isinstance(value, dict):
                value = value.get(lang)
            return value

    @property
    def deletable(self):
        if self.required:
            return False
        if not self.ignoreContext:
            value = self.current_value
        else:
            value = self.value
        return bool(value)

    def get_human_size(self, value):
        return get_human_size(value, self.request)

    def get_thumbnail(self, geometry='128x128'):
        thumbnails = IThumbnails(self.current_value, None)
        if thumbnails is not None:
            display = thumbnails.get_thumbnail(geometry)
            if display is not None:
                dc = IZopeDublinCore(display, None)
                if dc is not None:
                    timestamp = dc.modified.timestamp()
                else:
                    timestamp = self.timestamp
                return '{0}?_={1}'.format(absolute_url(display, self.request), timestamp)
        name, ext = os.path.splitext(self.current_value.filename)
        return '/--static--/myams/img/mimetypes/{0}'.format(EXTENSIONS_THUMBNAILS.get(ext, 'unknown.png'))

    def get_thumbnail_target(self):
        value = self.current_value
        if value is not None:
            registry = self.request.registry
            view = registry.queryMultiAdapter((value, self.request), IView, name='preview.html')
            if view is not None:
                return absolute_url(value, self.request, 'preview.html')


@adapter_config(context=(IFileField, IFormLayer), provides=IFieldWidget)
def FileFieldWidget(field, request):
    return FieldWidget(field, FileWidget(request))


#
# Image widgets
#

@widgettemplate_config(mode='input', template='templates/media-input.pt', widget=IMediaWidget, layer=IFormLayer)
@widgettemplate_config(mode='display', template='templates/media-display.pt', widget=IMediaWidget, layer=IFormLayer)
@implementer_only(IMediaWidget)
class MediaWidget(FileWidget):
    """Media file widget"""


@adapter_config(context=(IMediaField, IFormLayer), provides=IFieldWidget)
def MediaFieldWidget(field, request):
    return FieldWidget(field, MediaWidget(request))


@implementer_only(IThumbnailMediaWidget)
class ThumbnailMediaWidget(MediaWidget):
    """Media widget with thumbnail images selection"""


@adapter_config(context=(IThumbnailMediaField, IFormLayer), provides=IFieldWidget)
def ThumbnailMediaFieldWidget(field, request):
    return FieldWidget(field, ThumbnailMediaWidget(request))
