#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import logging
import re

import transaction
from BTrees import OOBTree
from persistent.dict import PersistentDict
from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.interface import Interface, alsoProvides
from zope.lifecycleevent import IObjectRemovedEvent, ObjectAddedEvent, ObjectCreatedEvent, \
    ObjectRemovedEvent
from zope.location import locate
from zope.traversing.interfaces import ITraversable

from pyams_file.file import FileFactory
from pyams_file.image import render_image
from pyams_file.interfaces import IFileModifiedEvent, IImage, IMediaFile, IThumbnailFile, \
    IThumbnailer, IThumbnails, IWatermarker
from pyams_utils.adapter import ContextAdapter, ContextRequestViewAdapter, adapter_config, \
    get_annotation_adapter
from pyams_utils.interfaces.tales import ITALESExtension
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request


__docformat__ = 'restructuredtext'


logger = logging.getLogger('PyAMS (file)')


THUMBNAIL_ANNOTATIONS_KEY = 'pyams_file.image.thumbnails'
THUMBNAIL_GEOMETRY_KEY = 'pyams_file.image.geometry'

THUMB_WIDTH = re.compile('^(?:\w+\:)?w([0-9]+)$')
THUMB_HEIGHT = re.compile('^(?:\w+\:)?h([0-9]+)$')
THUMB_SIZE = re.compile('^(?:\w+\:)?([0-9]+)x([0-9]+)$')


@adapter_config(context=IImage, provides=IThumbnails)
class ImageThumbnailAdapter(object):
    """Image thumbnails adapter"""

    def __init__(self, image):
        self.image = image
        self.thumbnails = get_annotation_adapter(image, THUMBNAIL_ANNOTATIONS_KEY, OOBTree.OOBTree,
                                                 notify=False, locate=False)

    def get_image_size(self):
        return self.image.get_image_size()

    def get_thumbnail_size(self, thumbnail_name, forced=False):
        width, height = self.get_image_size()
        match = THUMB_WIDTH.match(thumbnail_name)
        if match:
            w = int(match.groups()[0])
            w_ratio = 1. * width / w
            h_ratio = 0.
        else:
            match = THUMB_HEIGHT.match(thumbnail_name)
            if match:
                h = int(match.groups()[0])
                w_ratio = 0.
                h_ratio = 1. * height / h
            else:
                match = THUMB_SIZE.match(thumbnail_name)
                if match:
                    groups = match.groups()
                    w = int(groups[0])
                    h = int(groups[1])
                    w_ratio = 1. * width / w
                    h_ratio = 1. * height / h
        if match:
            ratio = max(w_ratio, h_ratio)
            if not forced:
                ratio = max(ratio, 1.)
            return int(width / ratio), int(height / ratio)
        else:
            return None

    def get_geometry(self, selection_name):
        geometries = get_annotation_adapter(self.image, THUMBNAIL_GEOMETRY_KEY, default={})
        # get default geometry for custom thumbnails
        if ':' in selection_name:
            selection_name, options = selection_name.split(':', 1)
        else:
            selection_name = selection_name
            options = None
        if selection_name in geometries:
            return geometries[selection_name]
        registry = check_request().registry
        thumbnailer = registry.queryAdapter(self.image, IThumbnailer, name=selection_name)
        if thumbnailer is not None:
            return thumbnailer.get_default_geometry(options)

    def set_geometry(self, selection_name, geometry):
        geometries = get_annotation_adapter(self.image, THUMBNAIL_GEOMETRY_KEY, PersistentDict,
                                            notify=False, locate=False)
        geometries[selection_name] = geometry
        for current_thumbnail_name in list(self.thumbnails.keys()):
            if (current_thumbnail_name == selection_name) or \
                    current_thumbnail_name.startswith('{0}:'.format(selection_name)):
                self.delete_thumbnail(current_thumbnail_name)

    def clear_geometries(self):
        geometries = get_annotation_adapter(self.image, THUMBNAIL_GEOMETRY_KEY)
        if geometries is not None:
            for geometry_name in list(geometries.keys()):
                del geometries[geometry_name]

    def get_thumbnail_name(self, thumbnail_name, with_size=False):
        size = self.get_thumbnail_size(thumbnail_name)
        if size is not None:
            if with_size:
                return '{0}x{1}'.format(*size), size
            else:
                return '{0}x{1}'.format(*size)
        else:
            return None, None

    def get_selection(self, selection_name, format=None):
        logger.debug(">>> Requested thumbnail selection: {}".format(selection_name))
        if selection_name in self.thumbnails:
            return self.thumbnails[selection_name]
        geometry = self.get_geometry(selection_name)
        if geometry == IThumbnailer(self.image).get_default_geometry():
            return self.image
        else:
            registry = get_current_registry()
            thumbnailer = registry.queryAdapter(self.image, IThumbnailer, name=selection_name)
            if thumbnailer is not None:
                selection = thumbnailer.create_thumbnail(geometry, format)
                if selection is not None:
                    if isinstance(selection, tuple):
                        selection, format = selection
                    else:
                        format = 'jpeg'
                    selection = FileFactory(selection)
                    alsoProvides(selection, IThumbnailFile)
                    registry.notify(ObjectCreatedEvent(selection))
                    self.thumbnails[selection_name] = selection
                    selection_size = selection.get_image_size()
                    locate(selection, self.image,
                           '++thumb++{0}:{1}x{2}.{3}'.format(selection_name,
                                                             selection_size[0],
                                                             selection_size[1],
                                                             format))
                    logger.debug("  > Generated thumbnail selection: {}".format(selection.__name__))
                    registry.notify(ObjectAddedEvent(selection))
                    return selection

    def get_thumbnail(self, thumbnail_name, format=None, watermark=None):
        logger.debug(">>> Requested thumbnail: {}".format(thumbnail_name))
        # check for existing thumbnail
        if thumbnail_name in self.thumbnails:
            return self.thumbnails[thumbnail_name]
        # check for selection thumbnail, in "selection:size" format
        if ':' in thumbnail_name:
            selection_name, size = thumbnail_name.split(':', 1)
            selection = self.get_selection(selection_name)
            if selection is not None:
                thumbnails = IThumbnails(selection)
                return thumbnails.get_thumbnail(size)
        # check for matching one
        name, size = self.get_thumbnail_name(thumbnail_name, with_size=True)
        if name:
            if name in self.thumbnails:
                return self.thumbnails[name]
            # check for original image
            if size == self.get_image_size():
                return self.image
            # wee will look for default image thumbnailer
            thumbnailer_name = ''
            options = name
        else:
            if ':' in thumbnail_name:
                thumbnailer_name, options = thumbnail_name.split(':', 1)
            else:
                thumbnailer_name = thumbnail_name
                options = name = thumbnail_name
        # generate and store thumbnail
        registry = get_current_registry()
        thumbnailer = registry.queryAdapter(self.image, IThumbnailer, name=thumbnailer_name)
        if thumbnailer is not None:
            thumbnail_image = thumbnailer.create_thumbnail(options, format)
            if thumbnail_image is not None:
                if isinstance(thumbnail_image, tuple):
                    thumbnail_image, format = thumbnail_image
                else:
                    format = 'jpeg'
                # check watermark
                if watermark is not None:
                    watermarker = query_utility(IWatermarker)
                    if watermarker is not None:
                        thumbnail_image.seek(0)
                        thumbnail_image, format = watermarker.add_watermark(thumbnail_image,
                                                                            watermark)
                # create final image
                thumbnail_image = FileFactory(thumbnail_image)
                alsoProvides(thumbnail_image, IThumbnailFile)
                registry.notify(ObjectCreatedEvent(thumbnail_image))
                self.thumbnails[name] = thumbnail_image
                thumbnail_size = thumbnail_image.get_image_size()
                locate(thumbnail_image, self.image,
                       '++thumb++{0}{1}{2}x{3}.{4}'.format(thumbnailer_name,
                                                           ':' if thumbnailer_name else '',
                                                           thumbnail_size[0],
                                                           thumbnail_size[1],
                                                           format))
                logger.debug("  < Generated thumbnail: {}".format(thumbnail_image.__name__))
                registry.notify(ObjectAddedEvent(thumbnail_image))
                return thumbnail_image

    def delete_thumbnail(self, thumbnail_name):
        if thumbnail_name in self.thumbnails:
            thumbnail_image = self.thumbnails[thumbnail_name]
            registry = get_current_registry()
            registry.notify(ObjectRemovedEvent(thumbnail_image))
            del self.thumbnails[thumbnail_name]
            logger.debug(">>> Removed thumbnail: {}".format(thumbnail_name))

    def clear_thumbnails(self):
        [self.delete_thumbnail(thumbnail_name) for thumbnail_name in list(self.thumbnails.keys())]


@subscriber(IFileModifiedEvent, context_selector=IMediaFile)
@subscriber(IObjectRemovedEvent, context_selector=IMediaFile)
def handle_modified_image(event):
    """Clear thumbnails when an image is updated or removed"""
    thumbnails = IThumbnails(event.object, None)
    if thumbnails is not None:
        thumbnails.clear_geometries()
        thumbnails.clear_thumbnails()


@adapter_config(name='thumb', context=IImage, provides=ITraversable)
class ThumbnailTraverser(ContextAdapter):
    """++thumb++ namespace traverser"""

    def traverse(self, name, furtherpath=None):
        if '.' in name:
            thumbnail_name, format = name.rsplit('.', 1)
        else:
            thumbnail_name = name
            format = None
        thumbnails = IThumbnails(self.context)
        if ':' in thumbnail_name:
            selection_name, thumbnail_name = thumbnail_name.split(':', 1)
            selection = thumbnails.get_selection(selection_name, format)
            if selection is not None:
                transaction.commit()
                thumbnails = IThumbnails(selection)
        result = thumbnails.get_thumbnail(thumbnail_name, format)
        transaction.commit()
        return result


@adapter_config(name='thumbnails', context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class ThumbnailsExtension(ContextRequestViewAdapter):
    """extension:thumbnails(image) TALES extension

    This TALES extension returns the IThumbnails adapter of given image.
    """

    def render(self, context=None):
        if context is None:
            context = self.context
        return IThumbnails(context, None)


@adapter_config(name='thumbnail', context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class ThumbnailExtension(ContextRequestViewAdapter):
    """extension:thumbnail(image, width, height, css_class, img_class) TALES extension

    This TALES extension doesn't return an adapter but HTML code matching given image and dimensions.
    If image is a classic image, an "img" tag with source to thumbnail of required size is returned.
    If image in an SVG image, a "div" is returned containing whole SVG data of given image.
    """

    def render(self, context=None, width=None, height=None, css_class='', img_class='', alt=''):
        if context is None:
            context = self.context
        return render_image(context, width, height, self.request, css_class, img_class, True, alt)
