#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

import os

from ZODB.POSException import POSKeyError
from ZODB.utils import oid_repr, p64, repr_to_oid
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import ObjectRemovedEvent

from pyams_file.interfaces import IBlobReferenceManager
from pyams_utils.decorator import deprecated
from pyams_utils.registry import get_global_registry, get_utility, set_local_registry


def check_before_upgrade(root):
    """Register all missing files before upgrading

    Run a ZODB search before upgrading to last PyAMS_file package:

    /var/local/env/pyams$ ./bin/zodbsearch --config etc/zodb-relstorage.conf \
        --data pyams_file.file.File pyams_file.file.ImageFile pyams_file.file.SVGImageFile \
        pyams_file.file.VideoFile pyams_file.file.AudioFile 2>&1 \
        | grep found | awk '{print $7}' | sort -u | cut -d \" -f 2 > var/FILES.oids


    """
    set_local_registry(root.getSiteManager())

    db = getattr(root, '_p_jar')
    intids = get_utility(IIntIds)

    count = 0
    files = open('var/FILES.oids', 'r')
    for oid in files.readlines():
        file_oid = p64(int(oid, 16))
        file = db.get(file_oid)
        ref = intids.queryId(file)
        if ref is None:
            intids.register(file)
            count += 1
    return count


def remove_unused_fs_blobs(root):
    """Filesystem blobs removal script

    Find all blobs before running this command, and store their path to file:

        /var/local/env/pyams$ find /var/db/blobs -name "*.blob" | cut -d \/ -f 4- > var/BLOBS.fs

    """
    db = getattr(root, '_p_jar')
    blobs = open('var/BLOBS.fs', 'r')

    for filename in blobs.readlines():
        oid, tid = filename.rsplit('/', 1)
        blob_oid = repr_to_oid(oid.replace('/0x', '').encode())
        try:
            db.get(blob_oid)
        except POSKeyError:
            print("Removing file: {}".format(filename[:-1]))
            try:
                os.remove('var/db/blobs/{}'.format(filename[:-1]))
                os.remove('var/db/blobs/{}/.lock'.format(oid))
                os.removedirs('var/db/blobs/{}'.format(oid))
            except OSError:
                pass


@deprecated
def remove_unused_blobs(root):
    """This method is deprecated and may generate errors: DON'T USE IT ANYMORE"""
    registry = get_global_registry()
    set_local_registry(root.getSiteManager())
    references = get_utility(IBlobReferenceManager)

    db = getattr(root, '_p_jar')
    files = open('var/FILES.oids', 'r')

    for foid in files.readlines():
        if not foid:
            continue
        file_oid = p64(int(foid, 16))
        file = db.get(file_oid)
        blob = getattr(file, '_blob', None)
        if blob is not None:
            blob_oid = oid_repr(getattr(blob, '_p_oid'))
            if blob_oid not in references.refs:
                delattr(file, '_blob')
                file._blob = None
                registry.notify(ObjectRemovedEvent(file))
                file.__parent__ = None
                print("{!r}".format(file))
        else:
            registry.notify(ObjectRemovedEvent(file))
            file.__parent__ = None
            print("Empty file {!r}".format(file))
