#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

# import standard library

# import interfaces
from pyams_file.interfaces import IFile, IFileField, IMediaFile, IMediaField, IBaseImage, IImageField, IVideo, \
    IVideoField, IAudio, IAudioField, IThumbnailMediaField, IThumbnailImageField, IThumbnailVideoField, DELETED_FILE
from z3c.form.interfaces import NOT_CHANGED
from zope.schema.interfaces import WrongType, RequiredMissing

# import packages
from zope.interface import implementer
from zope.schema import Bytes


@implementer(IFileField)
class FileField(Bytes):
    """Custom field used to handle file-like properties"""

    schema = IFile

    def _validate(self, value):
        if value is DELETED_FILE:
            if self.required and not self.default:
                raise RequiredMissing
        elif value is NOT_CHANGED:
            return
        elif isinstance(value, tuple):
            try:
                filename, stream = value
                if not isinstance(filename, str):
                    raise WrongType(filename, str, '{0}.filename'.format(self.__name__))
                if not hasattr(stream, 'read'):
                    raise WrongType(stream, '<file-like object>', self.__name__)
            except ValueError:
                raise WrongType(value, tuple, self.__name__)
        elif not self.schema.providedBy(value):
            raise WrongType(value, self.schema, self.__name__)


@implementer(IMediaField)
class MediaField(FileField):
    """Custom field used to store media-like properties"""

    schema = IMediaFile


@implementer(IThumbnailMediaField)
class ThumbnailMediaField(MediaField):
    """Custom field used to store media properties with thumbnail selection"""


@implementer(IImageField)
class ImageField(MediaField):
    """Custom field used to handle image properties"""

    schema = IBaseImage


@implementer(IThumbnailImageField)
class ThumbnailImageField(ImageField):
    """Custom field used to handle images properties with square selection"""


@implementer(IVideoField)
class VideoField(MediaField):
    """Custom field used to store video file"""

    schema = IVideo


@implementer(IThumbnailVideoField)
class ThumbnailVideoField(VideoField):
    """Custom field used to store video properties with thumbnail selection"""


@implementer(IAudioField)
class AudioField(MediaField):
    """Custom field used to store audio file"""

    schema = IAudio
