#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form.interfaces import IFileWidget as IBaseFileWidget
from zope.annotation.interfaces import IAttributeAnnotatable
from zope.interface import Attribute, Interface, implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.schema import Bytes, BytesLine, Choice, Int, Text, TextLine
from zope.schema.interfaces import IBytes

from pyams_file import _


#
# Blobs references manager
#

class IBlobReferenceManager(Interface):
    """Blobs references manager

    This utility interface is used to manage references to blobs: each file contains a
    link to a ZODB "Blob" object which is used to store it's data; when a content is duplicated,
    all it's blobs references are updated but the blob itself is not duplicated to reduce space usage.
    As long as a blob data is not modified, the same content can be shared between several versions
    of a same content. So it's only when all references to a given blob have been removed that
    the blob file is deleted and can be garbaged collected.
    """

    def add_reference(self, blob, reference):
        """Add a reference to given blob"""

    def drop_reference(self, blob, reference):
        """Remove reference from given blob

        Blob is deleted if no more referenced.
        """


#
# Main file objects interfaces
#

class IFileModifiedEvent(IObjectModifiedEvent):
    """Modified file event interface"""


@implementer(IFileModifiedEvent)
class FileModifiedEvent(ObjectModifiedEvent):
    """Modified file event"""


class IFile(IAttributeAnnotatable):
    """File object interface"""

    content_type = BytesLine(title="Content type",
                             description="The content type identifies the type of content data",
                             required=False,
                             default=b'',
                             missing_value=b'')

    data = Bytes(title="Content data",
                 description="Actual file content",
                 required=False,
                 default=b'',
                 missing_value=b'')

    def get_size(self):
        """Returns the byte-size of object's data"""

    def get_blob(self, mode='r'):
        """Get Blob file associated with this object"""

    def add_blob_reference(self, reference):
        """Add a reference to file internal blob"""

    def free_blob(self):
        """Free blob associated with this object"""


class ITempFile(IFile):
    """Temporary file marker interface"""

    filename = Attribute("Virtual file name")

    modification_date = Attribute("Virtual file modification date")


class IMediaFile(IFile):
    """Multimedia file"""


class IBaseImage(IMediaFile):
    """Base image interface"""


class IImage(IBaseImage):
    """Image object interface"""

    def get_image_size(self):
        """Returns an (x, y) tuple describing image dimensions"""

    def resize(self, width, height):
        """Resize image to given dimensions"""

    def crop(self, x1, y1, x2, y2):
        """Crop image to given coordinates"""

    def rotate(self, angle=-90):
        """Rotate image, default to right"""


class ISVGImage(IBaseImage):
    """SVG file interface"""


class IResponsiveImage(Interface):
    """Responsive image marker interface"""


class IVideo(IMediaFile):
    """Video file interface"""


class IAudio(IMediaFile):
    """Audio file interface"""


class IFileInfo(Interface):
    """File extended information"""

    title = TextLine(title=_("Title"),
                     required=False)

    description = Text(title=_("Description"),
                       required=False)

    filename = TextLine(title=_("Save file as..."),
                        description=_("Name under which the file will be saved"),
                        required=False)

    language = Choice(title=_("Language"),
                      description=_("File's content language"),
                      vocabulary="PyAMS base languages",
                      required=False)


class IFileFieldContainer(IAttributeAnnotatable):
    """Marker interface for contents holding file properties"""


#
# Schema fields interfaces
#

class DELETED_FILE(object):
    def __repr__(self):
        return '<DELETED_FILE>'


DELETED_FILE = DELETED_FILE()


class IThumbnailField(Interface):
    """Generic field interface with thumbnail"""


class IFileField(IBytes):
    """File object field interface"""

    schema = Attribute("Required value schema")


class IMediaField(IFileField):
    """Media file object field interface"""


class IThumbnailMediaField(IMediaField, IThumbnailField):
    """Media object field with thumbnail interface"""


class IImageField(IMediaField):
    """Image file object field interface"""


class IThumbnailImageField(IImageField, IThumbnailField):
    """Image object field with thumbnail interface"""


class IVideoField(IMediaField):
    """Video file field interface"""


class IThumbnailVideoField(IVideoField, IThumbnailField):
    """Video object field with thumbnail interface"""


class IAudioField(IMediaField):
    """Audio file field interface"""


#
# Widgets interfaces
#

class IFileWidget(IBaseFileWidget):
    """File field widget"""


class IMediaWidget(IFileWidget):
    """Media file widget"""


class IThumbnailMediaWidget(IMediaWidget):
    """Media file widget with thumbnail selection"""


class IImageWidget(IMediaWidget):
    """Image field widget"""


class IThumbnailImageWidget(IImageWidget):
    """Image field widget with thumbnail selection"""


#
# Thumbnails interfaces
#

THUMBNAILERS_VOCABULARY_NAME = 'pyams_file.image.thumbnailers'


class IThumbnailFile(Interface):
    """Marker interface for thumbnails"""


class IThumbnailGeometry(Interface):
    """Image thumbnail geometry interface"""

    x1 = Int(title="Thumbnail position X1",
             required=True,
             min=0)

    y1 = Int(title="Thumbnail position Y1",
             required=True,
             min=0)

    x2 = Int(title="Thumbnail position X2",
             required=True,
             min=0)

    y2 = Int(title="Thumbnail position Y2",
             required=True,
             min=0)

    def is_empty(self):
        """Check if geometry is not empty"""


class IThumbnailer(Interface):
    """Interface of adapter used to generate image thumbnails"""

    label = Attribute("Thumbnail label")
    section = Attribute("Thumbnail section")
    weight = Attribute("Thumbnail weight")

    def get_default_geometry(self):
        """Get default thumbnail geometry"""

    def create_thumbnail(self, target, format=None):
        """Create thumbnail of the given source object

        Source can be any file which can provide thumbnails (image, video,
        PDF file...).
        Target, which defines thumbnail size, can be defined as a selection name
        ('pano', 'square', 'xs'...), as a geometry or as a (width, height) tuple.

        If the requested image is of a resolution higher than that of the original file,
        the resulting image resolution will be that of the original file.

        If format (JPEG, PNG...) is given, this will be the format of the generated
        thumbnail; otherwise the format will be those of the source image.
        """


class IThumbnails(Interface):
    """Image thumbnail interface

    Displays are images thumbnails generated 'on the fly' and stored into image
    annotations for future use
    """

    def get_image_size(self):
        """Get original image size"""

    def get_thumbnail_size(self, thumbnail_name, forced=False):
        """Get real size of the genrated thumbnail

        If forced is True, the generated thumbnail can be larger than the original
        source
        """

    def get_geometry(self, selection_name):
        """Get geometry of a given thumbnail"""

    def set_geometry(self, selection_name, geometry):
        """Set geometry for given thumbnail"""

    def clear_geometries(self):
        """Remove all stored geometries from object annotations"""

    def get_thumbnail_name(self, thumbnail_name, with_size=None):
        """Get matching name for the given thumbnail name or size"""

    def get_selection(self, selection_name, format=None):
        """Get image for given user selection"""

    def get_thumbnail(self, thumbnail_name, format=None, watermark=None):
        """Get requested thumbnail

        Display can be specified as:
        - a name matching a custom thumbnailer utility
        - a width, as wXXX where XXX is the requested image width
        - a height, as hYYY, where YYY is the requested image height
        - a size, as XXXxYYY
        """

    def delete_thumbnail(self, thumbnail_name):
        """Remove selected thumbnail from object annotations"""

    def clear_thumbnails(self):
        """Remove all thumbnails from object annotations"""


class IWatermarker(Interface):
    """Interface of utility used to add image watermark"""

    def add_watermark(self, image, watermark, position='scale', opacity=1, format=None):
        """Add watermark to given image"""


class IFileModifierForm(Interface):
    """File modifier form marker interface

    These forms have a custom context permission checker.
    """


class IThumbnailForm(IFileModifierForm):
    """Image thumbnail selection form marker interface"""
