#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import random
import re
import sys
from io import BytesIO

from PIL import Image, ImageFilter
from pyquery import PyQuery
from pyramid.renderers import render
from zope.component import getAdapters
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_file.file import ImageFile
from pyams_file.interfaces import IImage, IResponsiveImage, ISVGImage, IThumbnailGeometry, \
    IThumbnailer, IThumbnails, THUMBNAILERS_VOCABULARY_NAME
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.request import check_request
from pyams_utils.url import absolute_url
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from pyams_file import _


WEB_FORMATS = ('JPEG', 'PNG', 'GIF')
THUMB_SIZE = re.compile('^(?:\w+\:)?([0-9]+)x([0-9]+)$')


@implementer(IThumbnailGeometry)
class ThumbnailGeometry(object):
    """Image thumbnail geometry"""

    x1 = FieldProperty(IThumbnailGeometry['x1'])
    y1 = FieldProperty(IThumbnailGeometry['y1'])
    x2 = FieldProperty(IThumbnailGeometry['x2'])
    y2 = FieldProperty(IThumbnailGeometry['y2'])

    def __repr__(self):
        return '<ThumbnailGeometry: x1,y1={0.x1},{0.y1} - x2,y2={0.x2},{0.y2}>'.format(self)

    def __eq__(self, other):
        if IThumbnailGeometry.providedBy(other):
            return (self.x1 == other.x1) and \
                   (self.x2 == other.x2) and \
                   (self.y1 == other.y1) and \
                   (self.y2 == other.y2)
        else:
            return False

    def is_empty(self):
        return (self.x2 <= self.x1) or (self.y2 <= self.y1)


@adapter_config(context=IImage, provides=IThumbnailer)
class ImageThumbnailer(ContextAdapter):
    """Image thumbnailer adapter"""

    label = _("Default thumbnail")
    section = _("Default thumbnail")
    weight = 1

    def get_default_geometry(self, options=None):
        """Default thumbnail geometry"""
        geometry = ThumbnailGeometry()
        width, height = self.context.get_image_size()
        geometry.x1 = 0
        geometry.y1 = 0
        geometry.x2 = width
        geometry.y2 = height
        return geometry

    def create_thumbnail(self, target, format=None):
        # check thumbnail name
        if isinstance(target, str):
            width, height = tuple(map(int, target.split('x')))
        elif IThumbnailGeometry.providedBy(target):
            width = target.x2 - target.x1
            height = target.y2 - target.y1
        elif isinstance(target, tuple):
            width, height = target
        else:
            return None
        # check format
        blob = self.context.get_blob(mode='r')
        if blob is None:
            return None
        image = Image.open(blob)
        if not format:
            format = image.format
        format = format.upper()
        if format not in WEB_FORMATS:
            format = 'JPEG'
        # check image mode
        if image.mode == 'P':
            if format == 'JPEG':
                image = image.convert('RGB')
            else:
                image = image.convert('RGBA')
        # generate thumbnail
        new_image = BytesIO()
        image.resize((width, height), Image.ANTIALIAS) \
             .filter(ImageFilter.UnsharpMask(radius=0.5, percent=100, threshold=0)) \
             .save(new_image, format)
        return new_image, format.lower()


class ImageSelectionThumbnailer(ImageThumbnailer):
    """Image thumbnailer based on user selection"""

    section = _("Custom selections")

    def create_thumbnail(self, target, format=None):
        # get thumbnail size
        if isinstance(target, str):
            geometry = IThumbnails(self.context).get_geometry(target)
            match = THUMB_SIZE.match(target)
            if match:
                width, height = tuple(map(int, match.groups()))
            else:
                width = abs(geometry.x2 - geometry.x1)
                height = abs(geometry.y2 - geometry.y1)
        elif IThumbnailGeometry.providedBy(target):
            geometry = target
            width = abs(geometry.x2 - geometry.x1)
            height = abs(geometry.y2 - geometry.y1)
        elif isinstance(target, tuple):
            width, height = target
            geometry = self.get_default_geometry()
        else:
            return None
        # check format
        blob = self.context.get_blob(mode='r')
        if blob is None:
            return None
        image = Image.open(blob)
        if not format:
            format = image.format
        format = format.upper()
        if format not in WEB_FORMATS:
            format = 'JPEG'
        # check image mode
        if image.mode == 'P':
            if format == 'JPEG':
                image = image.convert('RGB')
            else:
                image = image.convert('RGBA')
        # generate thumbnail
        new_image = BytesIO()
        thumb_size = self.get_thumb_size(width, height, geometry)
        image.crop((geometry.x1, geometry.y1, geometry.x2, geometry.y2)) \
             .resize(thumb_size, Image.ANTIALIAS) \
             .filter(ImageFilter.UnsharpMask(radius=0.5, percent=100, threshold=0)) \
             .save(new_image, format)
        return new_image, format.lower()

    def get_thumb_size(self, width, height, geometry):
        return width, height


class ImageRatioThumbnailer(ImageSelectionThumbnailer):
    """Image thumbnailer with specific ratio"""

    ratio = (None, None)  # (width, height) ratio tuple

    def get_default_geometry(self, options=None):
        geometry = ThumbnailGeometry()
        width, height = self.context.get_image_size()
        thumb_max_height = width * self.ratio[1] / self.ratio[0]
        if thumb_max_height >= height:
            # image wider
            thumb_width = height * self.ratio[0] / self.ratio[1]
            geometry.x1 = round((width / 2) - (thumb_width / 2))
            geometry.y1 = 0
            geometry.x2 = round((width / 2) + (thumb_width / 2))
            geometry.y2 = height
        else:
            thumb_height = thumb_max_height
            geometry.x1 = 0
            geometry.y1 = round((height / 2) - (thumb_height / 2))
            geometry.x2 = width
            geometry.y2 = round((height / 2) + (thumb_height / 2))
        return geometry


@adapter_config(name='portrait', context=IImage, provides=IThumbnailer)
class ImagePortraitThumbnailer(ImageRatioThumbnailer):
    """Image portrait thumbnail adapter"""

    label = _("Portrait thumbnail")
    weight = 5

    ratio = (3, 4)


@adapter_config(name='square', context=IImage, provides=IThumbnailer)
class ImageSquareThumbnailer(ImageRatioThumbnailer):
    """Image square thumbnail adapter"""

    label = _("Square thumbnail")
    weight = 6

    ratio = (1, 1)


@adapter_config(name='pano', context=IImage, provides=IThumbnailer)
class ImagePanoThumbnailer(ImageRatioThumbnailer):
    """Image panoramic thumbnail adapter"""

    label = _("Panoramic thumbnail")
    weight = 7

    ratio = (16, 9)

    def get_thumb_size(self, width, height, geometry):
        thumb_size = abs(geometry.x2 - geometry.x1), abs(geometry.y2 - geometry.y1)
        w_ratio = 1. * width / thumb_size[0]
        h_ratio = 1. * height / thumb_size[1]
        ratio = min(w_ratio, h_ratio)
        return round(ratio * thumb_size[0]), round(ratio * thumb_size[1])


@adapter_config(name='banner', context=IImage, provides=IThumbnailer)
class ImageBannerThumbnailer(ImageRatioThumbnailer):
    """Image banner thumbnail adapter"""

    label = _("Banner thumbnail")
    weight = 8

    ratio = (5, 1)


class ResponsiveImageThumbnailer(ImageSelectionThumbnailer):
    """Responsive image thumbnailer"""

    section = _("Responsive selections")


@adapter_config(name='xs', context=IResponsiveImage, provides=IThumbnailer)
class XsImageThumbnailer(ResponsiveImageThumbnailer):
    """eXtra-Small responsive image thumbnailer"""

    label = _("Smartphone thumbnail")
    weight = 10


@adapter_config(name='sm', context=IResponsiveImage, provides=IThumbnailer)
class SmImageThumbnailer(ResponsiveImageThumbnailer):
    """SMall responsive image thumbnailer"""

    label = _("Tablet thumbnail")
    weight = 11


@adapter_config(name='md', context=IResponsiveImage, provides=IThumbnailer)
class MdImageThumbnailer(ResponsiveImageThumbnailer):
    """MeDium responsive image thumbnailer"""

    label = _("Medium screen thumbnail")
    weight = 12


@adapter_config(name='lg', context=IResponsiveImage, provides=IThumbnailer)
class LgImageThumbnailer(ResponsiveImageThumbnailer):
    """LarGe responsive image thumbnailer"""

    label = _("Large screen thumbnail")
    weight = 13


@vocabulary_config(name=THUMBNAILERS_VOCABULARY_NAME)
class ImageThumbnailsVocabulary(SimpleVocabulary):
    """Image thumbnails vocabulary"""

    def __init__(self, context=None):
        if not IImage.providedBy(context):
            context = ImageFile()
        request = check_request()
        translate = request.localizer.translate
        terms = []
        for name, adapter in sorted(getAdapters((context,), IThumbnailer),
                                    key=lambda x: x[1].weight):
            if not name:
                continue
            terms.append(SimpleTerm(name, title=translate(adapter.label)))
        super().__init__(terms)


#
# SVG utilities
#

def render_svg(image, width=None, height=None, request=None, css_class='', img_class='', alt=''):
    """Render SVG file"""
    options = {}
    if width or height:
        options['style'] = 'width: {0}{1}; height: {2}{3};'.format(
            width, 'px' if isinstance(width, int) else '',
            height, 'px' if isinstance(height, int) else '')
    else:
        options['style'] = ''
    options['css_class'] = css_class
    if alt or img_class:
        svg = PyQuery(image.data)
        if alt:
            g = PyQuery('<g></g>')
            g.append(PyQuery('<title />').text(alt))
            for child in svg.children():
                g.append(child)
            svg.empty().append(g)
        if img_class:
            svg.attr('class', img_class)
        options['svg'] = svg.outer_html()
    else:
        options['svg'] = image.data
    return render('templates/svg-render.pt', options, request)


def render_img(image, width=None, height=None, request=None,
               css_class='', img_class='', timestamp=False, alt=''):
    """Render image thumbnail"""
    thumbnail = None
    thumbnails = IThumbnails(image, None)
    if thumbnails is not None:
        if width and height:
            thumbnail = thumbnails.get_thumbnail('{0}x{1}'.format(width, height))
        elif width and (width != 'auto'):
            thumbnail = thumbnails.get_thumbnail('w{0}'.format(width))
        elif height and (height != 'auto'):
            thumbnail = thumbnails.get_thumbnail('h{0}'.format(height))
    if thumbnail is None:
        thumbnail = image
    if request is None:
        request = check_request()
    url = absolute_url(thumbnail, request)
    if timestamp:
        dc = IZopeDublinCore(thumbnail, None)
        if dc is None:
            timestamp = random.randint(0, sys.maxsize)
        else:
            timestamp = dc.modified.timestamp()
        url += '?_={0}'.format(timestamp)
    result = '<img src="{0}" class="{1}" alt="{2}" />'.format(url, img_class, alt)
    if css_class:
        result = '<div class="{0}">{1}</div>'.format(css_class, result)
    return result


def render_image(image, width=None, height=None, request=None,
                 css_class='', img_class='', timestamp=False, alt=''):
    """Render image"""
    if ISVGImage.providedBy(image):
        return render_svg(image, width, height, request, css_class, img_class, alt)
    else:
        return render_img(image, width, height, request, css_class, img_class, timestamp, alt)
