#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import IInternalLink, InternalReferenceMixin
from pyams_content.shared.logo import IWfLogo
from pyams_content.shared.view.portlet.interfaces import IViewItemsPortletSettings
from pyams_default_theme.features.search.portlet import HIDDEN_HEADER_DISPLAY, \
    START_HEADER_DISPLAY
from pyams_default_theme.shared.view.portlet.interfaces import \
    IViewItemsPortletHorizontalRendererSettings, IViewItemsPortletSummaryRendererSettings, \
    IViewItemsPortletVerticalRendererSettings
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.text import get_text_start
from pyams_utils.url import canonical_url, relative_url


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


class BaseViewItemsPortletRenderer(PortletRenderer):
    """Base view items portlet renderer"""

    @staticmethod
    def is_internal_link(link):
        return IInternalLink.providedBy(link)

    def get_url(self, target):
        if IWfLogo.providedBy(target):
            if target.target is None:
                return target.url
            target = target.target
        if self.settings.force_canonical_url:
            return canonical_url(target, self.request)
        return relative_url(target, self.request)

    def render(self, template_name=''):
        result = super(BaseViewItemsPortletRenderer, self).render(template_name)
        if self.settings.first_page_only:
            start = int(self.request.params.get('start', 0))
            if start:
                return ''
        return result


#
# Default view portlet renderer
#

@factory_config(provided=IViewItemsPortletVerticalRendererSettings)
class ViewItemsPortletVerticalRendererSettings(InternalReferenceMixin, Persistent, Contained):
    """View items portlet vertical renderer settings"""

    display_illustrations = FieldProperty(
        IViewItemsPortletVerticalRendererSettings['display_illustrations'])
    thumbnails_selection = FieldProperty(
        IViewItemsPortletVerticalRendererSettings['thumbnails_selection'])
    display_breadcrumbs = FieldProperty(
        IViewItemsPortletVerticalRendererSettings['display_breadcrumbs'])
    display_tags = FieldProperty(IViewItemsPortletVerticalRendererSettings['display_tags'])
    paginate = FieldProperty(IViewItemsPortletVerticalRendererSettings['paginate'])
    page_size = FieldProperty(IViewItemsPortletVerticalRendererSettings['page_size'])
    reference = FieldProperty(IViewItemsPortletVerticalRendererSettings['reference'])
    link_label = FieldProperty(IViewItemsPortletVerticalRendererSettings['link_label'])


@adapter_config(context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/view-items-list.pt', layer=IPyAMSLayer)
class ViewItemsPortletVerticalRenderer(BaseViewItemsPortletRenderer):
    """View items portlet vertical renderer"""

    label = _("Simple vertical view")
    weight = 1

    settings_interface = IViewItemsPortletVerticalRendererSettings


#
# Small pictures portlet renderer
#

@factory_config(provided=IViewItemsPortletHorizontalRendererSettings)
class ViewItemsPortletHorizontalRendererSettings(Persistent, Contained):
    """View items portlet horizontal renderer settings"""

    thumbnails_selection = FieldProperty(
        IViewItemsPortletHorizontalRendererSettings['thumbnails_selection'])


@adapter_config(name='horizontal',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/view-items-horiz-list.pt', layer=IPyAMSLayer)
class ViewItemsPortletHorizontalRenderer(BaseViewItemsPortletRenderer):
    """View items portlet horizontal renderer"""

    label = _("Simple horizontal thumbnails view")
    weight = 2

    settings_interface = IViewItemsPortletHorizontalRendererSettings


#
# Summary-like portlet renderer
#

@factory_config(provided=IViewItemsPortletSummaryRendererSettings)
class ViewItemsPortletSummaryRendererSettings(Persistent, Contained):
    """View items portlet renderer summary renderer settings"""

    display_illustrations = FieldProperty(
        IViewItemsPortletSummaryRendererSettings['display_illustrations'])
    thumbnails_selection = FieldProperty(
        IViewItemsPortletSummaryRendererSettings['thumbnails_selection'])
    paginate = FieldProperty(IViewItemsPortletSummaryRendererSettings['paginate'])
    page_size = FieldProperty(IViewItemsPortletSummaryRendererSettings['page_size'])
    header_display_mode = FieldProperty(
        IViewItemsPortletSummaryRendererSettings['header_display_mode'])
    start_length = FieldProperty(IViewItemsPortletSummaryRendererSettings['start_length'])


@adapter_config(name='summary',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/view-items-summary.pt', layer=IPyAMSLayer)
class ViewItemsPortletSummaryRenderer(BaseViewItemsPortletRenderer):
    """View items portlet summary renderer"""

    label = _("Vertical panels view")
    weight = 3

    settings_interface = IViewItemsPortletSummaryRendererSettings

    def get_header(self, item):
        settings = self.renderer_settings
        display_mode = settings.header_display_mode
        if display_mode == HIDDEN_HEADER_DISPLAY:
            return ''
        header = II18n(item).query_attribute('header', request=self.request)
        if display_mode == START_HEADER_DISPLAY:
            header = get_text_start(header, settings.start_length)
        return header
