#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_content.component.illustration import ILinkIllustration
from pyams_content.shared.site.interfaces import IContentLink, IExternalContentLink
from pyams_default_theme.interfaces import IContentNavigationIllustration, IContentSummaryInfo
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import IRelativeURL

from pyams_default_theme import _


#
# Content link adapters
#

@adapter_config(context=(IContentLink, IPyAMSUserLayer),
                provides=IRelativeURL)
def content_link_relative_url(context, request):
    """Content link relative URL"""
    target = context.get_target()
    if target is not None:
        return request.registry.queryMultiAdapter((target, request), IRelativeURL)


@adapter_config(context=(IContentLink, IPyAMSUserLayer),
                provides=IContentSummaryInfo)
class ContentLinkSummaryAdapter(ContextRequestAdapter):
    """Content link summary adapter"""

    target_info = None

    def __init__(self, context, request):
        super(ContentLinkSummaryAdapter, self).__init__(context, request)
        target = self.target = self.context.target
        if target is not None:
            self.target_info = request.registry.queryMultiAdapter((target, request),
                                                                  IContentSummaryInfo)

    @property
    def title(self):
        title = II18n(self.context).query_attribute('navigation_title', request=self.request)
        if (not title) and (self.target_info is not None):
            title = self.target_info.title
        return title

    @property
    def header(self):
        if not self.context.show_header:
            return None
        header = II18n(self.context).query_attribute('navigation_header', request=self.request)
        if (not header) and (self.target_info is not None):
            header = self.target_info.header
        return header

    @property
    def button_title(self):
        title = None
        if self.target_info is not None:
            title = self.target_info.button_title
        if not title:
            title = _("Consult content")
        return self.request.localizer.translate(title)


@adapter_config(context=(IContentLink, IPyAMSUserLayer),
                provides=IContentNavigationIllustration)
def content_link_illustration_factory(context, request):
    """Content link illustration factory"""
    illustration = request.registry.queryAdapter(context, ILinkIllustration)
    if (illustration is not None) and illustration.has_data():
        return illustration
    target = context.target
    if target is not None:
        return request.registry.queryMultiAdapter((target, request),
                                                  IContentNavigationIllustration)


#
# External content link adapters
#

@adapter_config(context=(IExternalContentLink, IPyAMSUserLayer),
                provides=IRelativeURL)
class ExternalContentLinkRelativeUrl(ContextRequestAdapter):
    """External content link relative URL"""

    def get_url(self, display_context=None, view_name=None, query=None):
        return self.context.url


@adapter_config(context=(IExternalContentLink, IPyAMSUserLayer),
                provides=IContentSummaryInfo)
class ExternalContentLinkSummaryAdapter(ContextRequestAdapter):
    """External content link summary adapter"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('navigation_title', request=self.request)

    @property
    def header(self):
        return II18n(self.context).query_attribute('navigation_header', request=self.request)

    @property
    def button_title(self):
        return self.request.localizer.translate(_("Consult content"))


@adapter_config(context=(IExternalContentLink, IPyAMSUserLayer),
                provides=IContentNavigationIllustration)
def external_content_link_illustration_factory(context, request):
    """External content link illustration factory"""
    return request.registry.queryAdapter(context, ILinkIllustration)
