#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.httpexceptions import HTTPSeeOther

from pyams_content.shared.site import ISiteElementNavigation
from pyams_content.shared.site.interfaces import SITE_CONTAINER_REDIRECT_MODE, ISiteFolder
from pyams_default_theme.interfaces import IContentHeader, IContentNavigationTitle, \
    IContentSummaryInfo
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.views import PortalContextIndexPage, PortalContextPreviewPage
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.url import relative_url
from pyams_workflow.interfaces import IWorkflowPublicationInfo

from pyams_default_theme import _


@adapter_config(context=(ISiteFolder, IPyAMSUserLayer),
                provides=ISiteElementNavigation)
class SiteFolderNavigationAdapter(ContextRequestAdapter):
    """Site folder navigation adapter"""

    @property
    def visible(self):
        if not self.context.visible_in_list:
            return False
        return IWorkflowPublicationInfo(self.context).is_visible(self.request)


@adapter_config(context=(ISiteFolder, IPyAMSUserLayer),
                provides=IContentNavigationTitle)
def site_folder_navigation_title(context, request):
    """Site folder navigation title adapter"""
    i18n = II18n(context)
    title = i18n.query_attribute('navigation_title', request=request)
    if title is None:
        title = i18n.query_attribute('title', request=request)
    return title


@adapter_config(context=(ISiteFolder, IPyAMSUserLayer),
                provides=IContentHeader)
def site_folder_header_adapter(context, request):
    """Site folder header adapter"""
    return II18n(context).query_attribute('header', request=request)


@adapter_config(context=(ISiteFolder, IPyAMSUserLayer),
                provides=IContentSummaryInfo)
class SiteFolderSummaryAdapter(ContextRequestAdapter):
    """Site folder summary adapter"""

    @property
    def title(self):
        i18n = II18n(self.context)
        return i18n.query_attribute('navigation_title', request=self.request) or \
               i18n.query_attribute('title', request=self.request)

    @property
    def header(self):
        try:
            return II18n(self.context).query_attribute('header', request=self.request)
        except AttributeError:
            return None

    button_title = _("Consult folder")


@pagelet_config(name='', context=ISiteFolder, layer=IPyAMSUserLayer)
class SiteFolderIndexPage(PortalContextIndexPage):
    """Site folder index page"""

    def __call__(self, **kwargs):
        if self.context.navigation_mode == SITE_CONTAINER_REDIRECT_MODE:
            target = next(self.context.get_visible_items(self.request), None)
            if target is not None:
                return HTTPSeeOther(relative_url(target, request=self.request))
        return super().__call__(**kwargs)


@pagelet_config(name='preview.html', context=ISiteFolder, layer=IPyAMSUserLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SiteFolderPreviewPage(PortalContextPreviewPage):
    """Site folder preview page"""

    def __call__(self, **kwargs):
        if self.context.navigation_mode == SITE_CONTAINER_REDIRECT_MODE:
            target = next(self.context.get_visible_items(self.request), None)
            if target is not None:
                return HTTPSeeOther(relative_url(target, request=self.request,
                                                 view_name='preview.html'))
        return super().__call__(**kwargs)
