#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import Interface

from pyams_content.component.illustration import IIllustration
from pyams_content.component.theme import ITagsInfo
from pyams_content.features.share.interfaces import ISocialShareInfo
from pyams_content.shared.site import IWfSiteTopic
from pyams_content.shared.site.interfaces import ISiteContainer, ISiteFolder, ISiteManager
from pyams_default_theme.interfaces import IContentHeader
from pyams_default_theme.viewlet.breadcrumb import BreadcrumbsAdapter
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.interfaces.configuration import IConfiguration
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.interfaces.viewlet import IBreadcrumbs
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import ContentMeta, HTMLTagMeta, PropertyMeta, SchemaMeta
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces.url import ICanonicalURL
from pyams_utils.registry import get_utility
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, canonical_url
from pyams_workflow.interfaces import IWorkflowPublicationInfo


@adapter_config(context=(ISiteManager, IPyAMSUserLayer),
                provides=IContentHeader)
def site_manager_header_adapter(context, request):
    """Default shared content header adapter"""
    return II18n(context).query_attribute('header', request=request)


@adapter_config(name='title',
                context=(ISiteManager, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
@adapter_config(name='title',
                context=(ISiteFolder, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class SiteContainerTitleMetasAdapter(ContextRequestViewAdapter):
    """Site container metas adapter"""

    order = 1

    def get_metas(self):
        context = self.context
        request = self.request
        i18n = II18n(context)
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language
        title = i18n.query_attribute('title', lang=lang, request=request)
        yield HTMLTagMeta('title', title or '----')
        description = i18n.query_attribute('description', lang=lang, request=request)
        if (not description) and hasattr(context, 'header'):
            description = i18n.query_attribute('header', lang=lang, request=request)
        if description:
            yield ContentMeta('description', description)


#
# Site container Opengraph metas
#

@adapter_config(name='opengraph',
                context=(ISiteManager, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
@adapter_config(name='opengraph',
                context=(ISiteFolder, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class SiteManagerOpengraphMetasAdapter(ContextRequestViewAdapter):
    """Site manager Opengraph metas adapter"""

    order = 15

    def get_metas(self):
        context = self.context
        request = self.request
        i18n = II18n(context)
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language

        description = i18n.query_attribute('description', lang=lang, request=request)
        if (not description) and hasattr(context, 'header'):
            description = i18n.query_attribute('header', request=request)

        # main properties
        yield PropertyMeta('og:type', 'article')
        yield PropertyMeta('og:title', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield PropertyMeta('og:description', description)

        # URL and site name
        yield PropertyMeta('og:url', canonical_url(context, request))
        configuration = IConfiguration(request.root)
        yield PropertyMeta('og:site_name', configuration.title)

        # author account
        share_info = ISocialShareInfo(request.root, None)
        if share_info.fb_account:
            yield PropertyMeta('article:author', share_info.fb_account)

        # workflow informations
        pub_info = IWorkflowPublicationInfo(context, None)
        if pub_info is not None:
            if pub_info.first_publication_date:
                yield PropertyMeta('article:published_time', tztime(pub_info.first_publication_date).isoformat())
            if pub_info.publication_expiration_date:
                yield PropertyMeta('article:expiration_time', tztime(pub_info.publication_expiration_date).isoformat())

        # tags
        tags = ITagsInfo(context, None)
        if tags is not None:
            for tag in tags.tags or ():
                yield PropertyMeta('article:tag', tag.label)

        # illustration properties
        registry = request.registry
        thumbnail = None
        alt = None
        illustration = registry.queryAdapter(context, IIllustration, name='link')
        if (illustration is None) or (not illustration.has_data()):
            illustration = registry.queryAdapter(context, IIllustration)
        if (illustration is not None) and illustration.has_data():
            data = II18n(illustration).query_attribute('data', lang=lang, request=request)
            thumbnail = IThumbnails(data).get_thumbnail('800x600')
            yield PropertyMeta('og:image', absolute_url(thumbnail, request))
            if request.scheme == 'https':
                yield PropertyMeta('og:image:secure_url', absolute_url(thumbnail, request))
            else:
                yield PropertyMeta('og:image:url', absolute_url(thumbnail, request))
            yield PropertyMeta('og:image:type', thumbnail.content_type)
            image_size = thumbnail.image_size
            yield PropertyMeta('og:image:width', image_size[0])
            yield PropertyMeta('og:image:height', image_size[1])
            alt = II18n(illustration).query_attribute('alt_title', lang=lang, request=request)
            if alt:
                yield PropertyMeta('og:image:alt', alt)

        # locales properties
        yield PropertyMeta('og:locale', lang)
        for other_lang in II18nManager(context).languages or ():
            if other_lang != lang:
                yield PropertyMeta('og:locale:alternate', other_lang)

        # twitter properties
        if share_info is not None:
            if share_info.fb_app_id:
                yield PropertyMeta('fb:app_id', share_info.fb_app_id)
            if share_info.twitter_account:
                yield ContentMeta('twitter:site', share_info.twitter_account)
                yield ContentMeta('twitter:creator', share_info.twitter_creator_account or share_info.twitter_account)
        yield ContentMeta('twitter:title', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield ContentMeta('twitter:description', description)
        if thumbnail is not None:
            yield ContentMeta('twitter:card', 'summary_large_image')
            yield ContentMeta('twitter:image', absolute_url(thumbnail, request))
            if alt:
                yield ContentMeta('twitter:image:alt', alt)
        else:
            yield ContentMeta('twitter:card', 'summary')

        # Schema.org properties
        yield SchemaMeta('name', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield SchemaMeta('description', description)
        if thumbnail is not None:
            yield SchemaMeta('image', absolute_url(thumbnail, request))


@adapter_config(context=(IWfSiteTopic, IPyAMSUserLayer),
                provides=ICanonicalURL)
class SiteTopicCanonicalUrlAdapter(ContextRequestAdapter):
    """Site topic canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        parent = get_parent(self.context, ISiteContainer)
        return absolute_url(parent, self.request,
                            '+/{0}::{1}{2}{3}'.format(
                                ISequentialIdInfo(self.context).get_base_oid().strip(),
                                self.context.content_url,
                                '/{}'.format(view_name) if view_name else '.html',
                                '?{}'.format(query) if query else ''))


@adapter_config(context=(IWfSiteTopic, IPyAMSUserLayer, Interface),
                provides=IBreadcrumbs)
class SiteTopicBreadcrumbsAdapter(BreadcrumbsAdapter):
    """Site topic breadcrumbs adapter"""

    @property
    def items(self):
        parent = self.context.__parent__
        yield from self.get_items(parent)
