#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty

from pyams_content.features.renderer.interfaces import IContentRenderer
from pyams_content.shared.logo.interfaces import ILogosParagraph
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_default_theme.shared.logo.interfaces import DISABLED_LINK, \
    ILogosParagraphRendererSettings, \
    INTERNAL_FIRST
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.url import canonical_url, relative_url


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Logos paragraph default renderer
#

LOGOS_PARAGRAPH_RENDERER_SETTINGS_KEY = 'pyams_content.logos.renderer:default'


@factory_config(ILogosParagraphRendererSettings)
class LogosParagraphDefaultRendererSettings(Persistent, Location):
    """Logos paragraph default renderer settings"""

    target_priority = FieldProperty(ILogosParagraphRendererSettings['target_priority'])
    force_canonical_url = FieldProperty(ILogosParagraphRendererSettings['force_canonical_url'])


@adapter_config(context=ILogosParagraph,
                provides=ILogosParagraphRendererSettings)
def logos_paragraph_default_renderer_settings_factory(context):
    """Logos paragraph default renderer settings factory"""
    return get_annotation_adapter(context, LOGOS_PARAGRAPH_RENDERER_SETTINGS_KEY,
                                  ILogosParagraphRendererSettings)


@adapter_config(name='default',
                context=(ILogosParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/logos-default.pt', layer=IPyAMSLayer)
class LogosParagraphDefaultRenderer(BaseContentRenderer):
    """Logos paragraph default renderer"""

    label = _("Default logos renderer")

    i18n_context_attrs = ('title', )

    settings_interface = ILogosParagraphRendererSettings

    def get_internal_url(self, logo):
        target = logo.target
        if target is not None:
            if self.settings.force_canonical_url:
                url_getter = canonical_url
            else:
                url_getter = relative_url
            return url_getter(target, request=self.request)
        return None

    @staticmethod
    def get_external_url(logo):
        return logo.url

    def get_url(self, logo):
        priority = self.settings.target_priority
        if priority == DISABLED_LINK:
            return None
        order = [self.get_external_url, self.get_internal_url]
        if priority == INTERNAL_FIRST:
            order = reversed(order)
        for getter in order:
            result = getter(logo)
            if result is not None:
                return result
        return None
