#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form.browser.checkbox import SingleCheckBoxWidget
from z3c.form.interfaces import IFieldWidget
from z3c.form.widget import FieldWidget
from zope.schema.interfaces import IBool, IDate, IDecimal, IList

from pyams_content.shared.form.interfaces import IFormFieldDataConverter
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.date import EXT_DATE_FORMAT, format_date


__docformat__ = 'restructuredtext'


@adapter_config(context=(IBool, IPyAMSUserLayer), provides=IFieldWidget)
def BooleanFieldWidget(field, request):
    return FieldWidget(field, SingleCheckBoxWidget(request))


@adapter_config(context=(IBool, IPyAMSUserLayer), provides=IFormFieldDataConverter)
class BooleanFieldDataConverter(ContextRequestAdapter):
    """Boolean field user data converter"""

    def convert(self, value):
        return self.context.description if value else '--'


@adapter_config(context=(IDate, IPyAMSUserLayer), provides=IFormFieldDataConverter)
class DateFieldDataConverter(ContextRequestAdapter):
    """Date field user data converter"""

    @staticmethod
    def convert(value):
        return format_date(value, EXT_DATE_FORMAT) if value else '--'


@adapter_config(context=(IDecimal, IPyAMSUserLayer), provides=IFormFieldDataConverter)
class DecimalFieldDataConverter(ContextRequestAdapter):
    """Decimal field user data converter"""

    @staticmethod
    def convert(value):
        return str(value) if value else '--'


@adapter_config(context=(IList, IPyAMSUserLayer), provides=IFormFieldDataConverter)
class ListFieldDataConverter(ContextRequestAdapter):
    """List field user data converter"""

    @staticmethod
    def convert(value):
        return ' / '.join(value) if value is not None else '--'
