#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import requests
from pyramid.csrf import get_csrf_token
from z3c.form import button, field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.interfaces import HIDDEN_MODE
from zope.interface import Interface, Invalid, alsoProvides
from zope.schema import Bool, TextLine

from pyams_content.shared.form import IFormFieldContainer, IFormFieldContainerTarget, \
    IFormsManager, IWfForm
from pyams_content.shared.form.interfaces import IFormFieldContainerInputForm, \
    IFormFieldDataConverter
from pyams_default_theme.component.paragraph.interfaces import IParagraphContainerPortletRenderer
from pyams_default_theme.interfaces import IContentTitle
from pyams_default_theme.layer import IPyAMSDefaultLayer
from pyams_default_theme.shared.common.interfaces import IPublicURL
from pyams_form.form import AddForm
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.views import PortalContextIndexPage
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces import PUBLIC_PERMISSION, VIEW_PERMISSION
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.text import text_to_html
from pyams_utils.traversing import get_parent
from pyams_utils.url import relative_url
from pyams_viewlet.viewlet import ViewContentProvider


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


CSRF_FIELD_NAME = 'csrf_token'
RECAPTCHA_FIELD_NAME = 'g-recaptcha-response'
RGPD_CONSENT_FIELD_NAME = 'rgpd_consent'

FORM_SUBMIT_ERROR_STATUS = _("Can't submit form.")
MISSING_TOKEN_ERROR = _("Missing recaptcha token!")
INVALID_TOKEN_ERROR = _("Can't verify recaptcha token! Are you a robot?")


@adapter_config(context=(IWfForm, IPyAMSDefaultLayer),
                provides=IContentTitle)
def form_title_adapter(context, request):
    """Form title adapter"""
    i18n = II18n(context)
    return i18n.query_attribute('alt_title', request=request) or \
           i18n.query_attribute('title', request=request)


@adapter_config(context=(Interface, IPyAMSDefaultLayer),
                provides=IFormFieldContainerInputForm)
class FormFieldContainerInputForm(AddForm):
    """Form fields container display form"""

    formErrorsMessage = FORM_SUBMIT_ERROR_STATUS

    @property
    def legend(self):
        return II18n(self.context).query_attribute('user_title', request=self.request)

    @property
    def edit_permission(self):
        if self.context.auth_only:
            return VIEW_PERMISSION
        else:
            return PUBLIC_PERMISSION

    def get_form_action(self):
        return relative_url(self.context, self.request, view_name='submit.html')

    @property
    def fields(self):

        def get_fields():
            form = self.context
            token = TextLine(title=_("CSRF token"), required=True)
            token.__name__ = CSRF_FIELD_NAME
            yield token
            captcha_settings = form.get_captcha_settings()
            if captcha_settings.get('use_captcha'):
                captcha = TextLine(title=_("Captcha"), required=True)
                captcha.__name__ = RECAPTCHA_FIELD_NAME
                yield captcha
            yield from IFormFieldContainer(form).get_fields()
            if self.context.rgpd_consent:
                consent = Bool(title=' ',
                               description=II18n(form).query_attribute('rgpd_warning',
                                                                       request=self.request),
                               required=True,
                               default=False)
                consent.__name__ = RGPD_CONSENT_FIELD_NAME
                yield consent

        fields = field.Fields(*tuple(get_fields()))
        if self.context.rgpd_consent:
            fields[RGPD_CONSENT_FIELD_NAME].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    def updateActions(self):
        super(FormFieldContainerInputForm, self).updateActions()
        if 'submit' in self.actions:
            self.actions['submit'].title = II18n(self.context).query_attribute('submit_label',
                                                                               request=self.request)

    def updateWidgets(self, prefix=None):
        super(FormFieldContainerInputForm, self).updateWidgets(prefix)
        request = self.request
        for widget in self.widgets.values():
            if widget.field.__name__ == CSRF_FIELD_NAME:
                widget.name = CSRF_FIELD_NAME
                widget.mode = HIDDEN_MODE
                widget.value = get_csrf_token(request)
            elif widget.field.__name__ == RECAPTCHA_FIELD_NAME:
                widget.name = RECAPTCHA_FIELD_NAME
                widget.mode = HIDDEN_MODE
            elif widget.field.__name__ == RGPD_CONSENT_FIELD_NAME:
                widget.name = RGPD_CONSENT_FIELD_NAME
                widget.description = ' '
                widget.required = False
                user_rights = II18n(self.context).query_attribute('rgpd_user_rights',
                                                                  request=request)
                widget.after_widget_notice = '<div><br />{0}</div>'.format(
                    text_to_html(user_rights, 'oid_to_href'))
                widget.object_data = {
                    'ams-validate-messages': {
                        'required': request.localizer.translate(
                            _("You can't submit this form without accepting data usage rules."))
                    }
                }
                alsoProvides(widget, IObjectData)
            else:
                field = IFormFieldContainer(self.context).get(widget.field.__name__)
                if field is not None:
                    # switch label and description for boolean fields
                    if field.field_type == 'bool':
                        widget.description = ' '
                        widget.label = II18n(field).query_attribute('label',
                                                                    request=request)
                    elif field.field_type == 'choice':
                        if field.placeholder:
                            widget.prompt = True
                            widget.promptMessage = field.placeholder
                    else:
                        widget.placeholder = field.placeholder

    @button.buttonAndHandler('title', name='submit')
    def update_content(self, action):
        request = self.request
        form = IWfForm(self.context)
        handler = form.query_handler()
        if handler is not None:
            data, errors = self.extractData()
            if errors:
                self.status = request.localizer.translate(self.formErrorsMessage)
            else:
                # remove custom data fields from handler data
                if CSRF_FIELD_NAME in data:
                    del data[CSRF_FIELD_NAME]
                captcha_settings = form.get_captcha_settings()
                if captcha_settings.get('use_captcha'):
                    if RECAPTCHA_FIELD_NAME not in data:
                        self.add_error(Invalid(MISSING_TOKEN_ERROR), RECAPTCHA_FIELD_NAME)
                        return
                    manager = get_parent(form, IFormsManager)
                    proxy_url = manager.get_proxy_url(request)
                    proxies = {'https': proxy_url} if proxy_url else {}
                    recaptcha_verify_api = \
                        request.registry.settings.get('pyams.recaptcha.verify')
                    if not recaptcha_verify_api:
                        recaptcha_verify_api = 'https://www.google.com/recaptcha/api/siteverify'
                    verify = requests.post(recaptcha_verify_api, {
                        'secret': captcha_settings.get('server_key'),
                        'response': data[RECAPTCHA_FIELD_NAME]
                    }, proxies=proxies).json()
                    if not verify['success']:
                        self.add_error(INVALID_TOKEN_ERROR, RECAPTCHA_FIELD_NAME)
                        return
                    del data[RECAPTCHA_FIELD_NAME]
                # convert form data
                user_data = data.copy()
                for form_field in IFormFieldContainer(form).get_fields():
                    converter = request.registry.queryMultiAdapter((form_field, request),
                                                                   IFormFieldDataConverter)
                    if converter is not None:
                        user_data[form_field.__name__] = converter.convert(
                            data.get(form_field.__name__))
                    if not user_data[form_field.__name__]:
                        user_data[form_field.__name__] = '--'
                output = handler.handle(form, data, user_data)
                if output:
                    request.annotations['form.output'] = output


@adapter_config(context=(IFormFieldContainerTarget, IPyAMSLayer, FormFieldContainerInputForm),
                provides=IFormHelp)
class FormFieldContainerDisplayFormHelp(FormHelp):
    """Form field container display form help adapter"""

    def __new__(cls, context, request, view):
        if not context.header:
            return None
        return FormHelp.__new__(cls)

    @property
    def message(self):
        return II18n(self.context).query_attribute('form_header', request=self.request) or ''

    message_format = 'text'


@pagelet_config(name='submit.html', context=IWfForm, layer=IPyAMSDefaultLayer)
class FormSubmitPage(PortalContextIndexPage):
    """Form submit page"""

    input_form = None

    def __init__(self, context, request):
        super(FormSubmitPage, self).__init__(context, request)
        self.input_form = request.registry.getMultiAdapter((context, self.request),
                                                           IFormFieldContainerInputForm)

    def update(self):
        super(FormSubmitPage, self).update()
        self.input_form.update()


@adapter_config(name='submit.html', context=(IWfForm, IPyAMSDefaultLayer, Interface),
                provides=IParagraphContainerPortletRenderer)
@template_config(template='templates/form-submit.pt', layer=IPyAMSDefaultLayer)
class FormSubmitPortletRenderer(ViewContentProvider):
    """Form submit message portlet renderer"""

    use_portlets_cache = False

    def render(self, template_name=''):
        form = self.view.input_form
        if form.widgets.errors:
            return form.render()
        return super().render(template_name)


@adapter_config(name='submit.html',
                context=(IWfForm, IPyAMSDefaultLayer),
                provides=IPublicURL)
class FormSubmitURLGetter(ContextRequestAdapter):
    """Form submit URL getter"""

    def get_url(self):
        return self.request.url
