#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.encode import urlencode, url_quote
from pyramid.url import QUERY_SAFE

from pyams_content.shared.common import ISharedContent, IWfSharedContent
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.url import absolute_url, canonical_url, relative_url


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer), provides=ICanonicalURL)
class WfSharedContentCanonicalUrlAdapter(ContextRequestAdapter):
    """Workflow managed shared content canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        if query:
            if isinstance(query, str):
                query = url_quote(query, QUERY_SAFE)
            else:
                query = urlencode(query, doseq=True)
        return absolute_url(self.request.root, self.request,
                            '+/{0}::{1}{2}{3}'.format(
                                ISequentialIdInfo(self.context).get_base_oid().strip(),
                                self.context.content_url,
                                '/{}'.format(view_name) if view_name else '.html',
                                '?{}'.format(query) if query else ''))


@adapter_config(context=(ISharedContent, IPyAMSUserLayer), provides=ICanonicalURL)
class SharedContentCanonicalURLAdapter(ContextRequestAdapter):
    """Shared content canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        version = self.context.visible_version
        if version is not None:
            return canonical_url(version, self.request, view_name, query)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer), provides=IRelativeURL)
class WfSharedContentRelativeUrlAdapter(ContextRequestAdapter):
    """Workflow managed shared content relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        if query:
            if isinstance(query, str):
                query = url_quote(query, QUERY_SAFE)
            else:
                query = urlencode(query, doseq=True)
        return absolute_url(display_context, self.request,
                            '+/{0}::{1}{2}{3}'.format(
                                ISequentialIdInfo(self.context).get_base_oid().strip(),
                                self.context.content_url,
                                '/{}'.format(view_name) if view_name else '.html',
                                '?{}'.format(query) if query else ''))


@adapter_config(context=(ISharedContent, IPyAMSUserLayer), provides=IRelativeURL)
class SharedContentRelativeURLAdapter(ContextRequestAdapter):
    """Shared content relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        version = self.context.visible_version
        if version is not None:
            return relative_url(version, self.request, display_context, view_name, query)
