#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface

from pyams_content.component.illustration import IIllustration, IIllustrationTarget
from pyams_content.component.theme import ITagsInfo
from pyams_content.features.share.interfaces import ISocialShareInfo
from pyams_content.shared.common import IWfSharedContent
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.interfaces.configuration import IConfiguration
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import ContentMeta, PropertyMeta, SchemaMeta
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, canonical_url
from pyams_workflow.interfaces import IWorkflowPublicationInfo


@adapter_config(name='opengraph', context=(IWfSharedContent, IPyAMSUserLayer, Interface), provides=IHTMLContentMetas)
class SharedContentOpengraphMetasAdapter(ContextRequestViewAdapter):
    """Shared content opengraph metas adapter"""

    order = 15

    def get_metas(self):
        context = self.context
        request = self.request
        i18n = II18n(context)
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language

        description = i18n.query_attribute('description', lang=lang, request=request) or \
                      i18n.query_attribute('header', lang=lang, request=request)

        # main properties
        yield PropertyMeta('og:type', 'article')
        yield PropertyMeta('og:title', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield PropertyMeta('og:description', description)

        # URL and site name
        yield PropertyMeta('og:url', canonical_url(context, request))
        configuration = IConfiguration(request.root)
        yield PropertyMeta('og:site_name', configuration.title)

        # author account
        share_info = ISocialShareInfo(request.root, None)
        if share_info.fb_account:
            yield PropertyMeta('article:author', share_info.fb_account)

        # workflow information
        dc = IZopeDublinCore(context, None)
        if (dc is not None) and dc.modified:
            yield PropertyMeta('article:modified_time', tztime(dc.modified).isoformat())
        pub_info = IWorkflowPublicationInfo(context, None)
        if pub_info is not None:
            if pub_info.first_publication_date:
                yield PropertyMeta('article:published_time', tztime(pub_info.first_publication_date).isoformat())
            if pub_info.publication_expiration_date:
                yield PropertyMeta('article:expiration_time', tztime(pub_info.publication_expiration_date).isoformat())

        # tags
        tags = ITagsInfo(context, None)
        if tags is not None:
            for tag in tags.tags or ():
                yield PropertyMeta('article:tag', tag.label)

        # illustration properties
        registry = request.registry
        illustration = None
        thumbnail = None
        alt = None
        target = context
        while target is not None:
            illustration = registry.queryAdapter(target, IIllustration, name='link')
            if (illustration is None) or (not illustration.has_data()):
                illustration = registry.queryAdapter(target, IIllustration)
            if (illustration is not None) and illustration.has_data():
                break
            target = get_parent(target, IIllustrationTarget, allow_context=False)
        if (target is not None) and (illustration is not None):
            data = II18n(illustration).query_attribute('data', lang=lang, request=request)
            if data:
                thumbnail = IThumbnails(data).get_thumbnail('800x600')
                yield PropertyMeta('og:image', absolute_url(thumbnail, request))
                if request.scheme == 'https':
                    yield PropertyMeta('og:image:secure_url', absolute_url(thumbnail, request))
                else:
                    yield PropertyMeta('og:image:url', absolute_url(thumbnail, request))
                yield PropertyMeta('og:image:type', thumbnail.content_type)
                image_size = thumbnail.image_size
                yield PropertyMeta('og:image:width', image_size[0])
                yield PropertyMeta('og:image:height', image_size[1])
                alt = II18n(illustration).query_attribute('alt_title', lang=lang, request=request)
                if alt:
                    yield PropertyMeta('og:image:alt', alt)

        # locales properties
        yield PropertyMeta('og:locale', lang)
        manager = II18nManager(context, None)
        if manager is not None:
            for other_lang in manager.languages or ():
                if other_lang != lang:
                    yield PropertyMeta('og:locale:alternate', other_lang)

        # twitter properties
        if share_info is not None:
            if share_info.fb_app_id:
                yield PropertyMeta('fb:app_id', share_info.fb_app_id)
            if share_info.twitter_account:
                yield ContentMeta('twitter:site', share_info.twitter_account)
                yield ContentMeta('twitter:creator', share_info.twitter_creator_account or share_info.twitter_account)
        yield ContentMeta('twitter:title', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield ContentMeta('twitter:description', description)
        if thumbnail is not None:
            yield ContentMeta('twitter:card', 'summary_large_image')
            yield ContentMeta('twitter:image', absolute_url(thumbnail, request))
            if alt:
                yield ContentMeta('twitter:image:alt', alt)
        else:
            yield ContentMeta('twitter:card', 'summary')

        # Schema.org properties
        yield SchemaMeta('name', i18n.query_attribute('title', lang=lang, request=request))
        if description:
            yield SchemaMeta('description', description)
        if thumbnail is not None:
            yield SchemaMeta('image', absolute_url(thumbnail, request))
