#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.exceptions import NotFound
from zope.interface import Interface
from zope.traversing.interfaces import ITraversable

from pyams_sequence.interfaces import ISequentialIntIds
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.registry import get_utility
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowVersions


@adapter_config(name='+', context=(Interface, IPyAMSUserLayer), provides=ITraversable)
@adapter_config(name='oid', context=(Interface, IPyAMSUserLayer), provides=ITraversable)
class OidTraverser(ContextRequestAdapter):
    """++oid++ traverser

    This traverser can be used to get a direct access to any content having an OID.
    The general URL syntax is "*/++oid++{oid}::{title}.html", where {oid} is the internal OID of the
    requested content, dans "title" it's "content URL" attribute.

    A shorter syntax, is now available: */+/{oid}::{title}.html
    """

    def traverse(self, name, furtherpath=None):
        if not name:
            raise NotFound()
        context = self.context
        request = self.request
        oid = name.split('::')[0]
        sequence = get_utility(ISequentialIntIds)
        reference = sequence.get_full_oid(oid)
        target = get_reference_target(reference)
        if target is not None:
            workflow = IWorkflow(target, None)
            if workflow is not None:
                versions = IWorkflowVersions(target, None)
                if versions is not None:
                    versions = versions.get_versions(workflow.visible_states, sort=True)
                    if versions:
                        target = versions[-1]
        if (target is not None) and not IWorkflowPublicationInfo(target).is_visible(request):
            target = None
        if target is not None:
            request.annotations[DISPLAY_CONTEXT] = context
            request.context = target
            return target
        raise NotFound()
