#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import Interface

from pyams_content.interfaces import IBaseContent
from pyams_content.shared.common import IBaseSharedTool
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_default_theme.interfaces import IContentDate, IContentHeader, IContentTag, \
    IContentTitle
from pyams_i18n.interfaces import II18n
from pyams_sequence.interfaces import IInternalReference
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import ContentMeta, HTMLTagMeta
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.date import SH_DATE_FORMAT, format_date
from pyams_utils.interfaces.tales import ITALESExtension
from pyams_utils.traversing import get_parent
from pyams_workflow.interfaces import IWorkflowPublicationInfo


__docformat__ = 'restructuredtext'


@adapter_config(name='content_type',
                context=(Interface, IPyAMSUserLayer, Interface),
                provides=ITALESExtension)
class PyAMSContentTypeTALESExtension(ContextRequestViewAdapter):
    """PyAMS content type TALES extension"""

    def render(self, context=None):
        if context is None:
            context = self.context
        content = IWfSharedContent(context, None)
        if content is not None:
            return self.request.localizer.translate(content.content_name)


@adapter_config(name='data_type',
                context=(Interface, IPyAMSUserLayer, Interface),
                provides=ITALESExtension)
class PyAMSContentDatatypeTALESExtension(ContextRequestViewAdapter):
    """PyAMS content data type TALES extension"""

    def render(self, context=None):
        if context is None:
            context = self.context
        content = IWfTypedSharedContent(context, None)
        if content is not None:
            data_type = content.get_data_type()
            if data_type is not None:
                return data_type.name


@adapter_config(name='title',
                context=(IWfSharedContent, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class SharedContentTitleMetasAdapter(ContextRequestViewAdapter):
    """Title metas adapter"""

    order = 1

    def get_metas(self):
        i18n = II18n(self.context)
        title = i18n.query_attribute('title', request=self.request)
        yield HTMLTagMeta('title', title or '----')
        description = i18n.query_attribute('description', request=self.request)
        if (not description) and self.context.handle_header:
            description = i18n.query_attribute('header', request=self.request)
        if description:
            yield ContentMeta('description', description)


@adapter_config(context=(IBaseContent, IPyAMSUserLayer),
                provides=IContentTitle)
def base_content_title_adapter(context, request):
    """Base content title adapter"""
    return II18n(context).query_attribute('title', request=request)


@adapter_config(context=(IInternalReference, IPyAMSUserLayer),
                provides=IContentTitle)
def internal_link_content_title_adapter(context, request):
    """Internal link content title adapter"""
    registry = request.registry
    return registry.queryMultiAdapter((context.target, request), IContentTitle)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer),
                provides=IContentHeader)
def shared_content_header_adapter(context, request):
    """Default shared content header adapter"""
    return II18n(context).query_attribute('header', request=request)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer),
                provides=IContentTag)
def shared_content_tag_adapter(context, request):
    """Default shared content tag adapter"""
    tool = get_parent(context, IBaseSharedTool)
    label = II18n(tool).query_attributes_in_order(('navigation_label', 'label'),
                                                  request=request)
    if label:
        return label
    translate = request.localizer.translate
    return translate(context.content_name)


@adapter_config(context=(IWfTypedSharedContent, IPyAMSUserLayer),
                provides=IContentTag)
def typed_shared_content_tag_adapter(context, request):
    """Typed shared content tag adapter"""
    data_type = context.get_data_type()
    if (data_type is not None) and data_type.display_as_tag:
        return II18n(data_type).query_attributes_in_order(('navigation_label', 'label'),
                                                          request=request)
    return shared_content_tag_adapter(context, request)


@adapter_config(name='pyams_tag',
                context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class PyAMSContentTagTALESExtension(ContextRequestViewAdapter):
    """PyAMS content tag TALES extension"""

    def render(self, context=None):
        if context is None:
            context = self.context
        return self.request.registry.queryMultiAdapter((context, self.request), IContentTag)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer),
                provides=IContentDate)
def shared_content_date_adapter(context, request):
    """Default shared content date adapter"""
    publication_info = IWorkflowPublicationInfo(context, None)
    if publication_info is not None:
        return format_date(publication_info.visible_publication_date,
                           format_string=SH_DATE_FORMAT, request=request)


@adapter_config(name='pyams_date',
                context=(Interface, Interface, Interface),
                provides=ITALESExtension)
class PyAMSContentDateTALESExtension(ContextRequestViewAdapter):
    """PyAMS content date TALES extension"""

    def render(self, context=None):
        if context is None:
            context = self.context
        return self.request.registry.queryMultiAdapter((context, self.request), IContentDate)
