#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from datetime import datetime, timedelta

from pyramid.decorator import reify
from zope.interface import Interface

from pyams_content.component.calendar import ICalendarInfo
from pyams_content.shared.calendar import IWfCalendarEvent
from pyams_default_theme.shared.common.interfaces import ICustomContentSpecificitiesRenderer
from pyams_skin.layer import IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.timezone import tztime
from pyams_viewlet.viewlet import ViewContentProvider

from pyams_default_theme import _


START_DATE_FORMATTER = _("on %d/%m/%Y, from %H:%M")
END_DATE_FORMATTER = _("to %H:%M")

MULTI_DATES_START_FORMATTER = _("from %d/%m/%Y at %H:%M")
MULTI_DATES_END_FORMATTER = _("to %d/%m/%Y at %H:%M")


@adapter_config(context=(IWfCalendarEvent, IPyAMSUserLayer, Interface),
                provides=ICustomContentSpecificitiesRenderer)
@template_config(template='templates/specificities.pt', layer=IPyAMSUserLayer)
class CalendarEventSpecificitiesPortletRenderer(ViewContentProvider):
    """Calendar event specificities portlet renderer"""

    @property
    def now(self):
        return tztime(datetime.utcnow())

    @reify
    def calendar_info(self):
        return ICalendarInfo(self.context)

    def format_datetime(self, start_datetime):
        start_datetime = tztime(start_datetime)
        duration = self.calendar_info.duration
        if not duration:
            return format_datetime(start_datetime)
        translate = self.request.localizer.translate
        end_datetime = start_datetime + timedelta(hours=duration)
        if start_datetime.date() == end_datetime.date():
            return '{} {}'.format(start_datetime.strftime(translate(START_DATE_FORMATTER)),
                                  end_datetime.strftime(translate(END_DATE_FORMATTER)))
        return '{} {}'.format(start_datetime.strftime(translate(MULTI_DATES_START_FORMATTER)),
                              end_datetime.strftime(translate(MULTI_DATES_END_FORMATTER)))
