#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.exceptions import NotFound
from pyramid.response import Response
from pyramid.view import render_view_to_response, view_config

from pyams_default_theme.shared.common.interfaces import IPublicURL
from pyams_sequence.interfaces import ISequentialIntIds
from pyams_sequence.reference import get_reference_target
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url, canonical_url
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowVersions


@view_config(route_name='oid_access')
def get_oid_access(request):
    """Get direct access to given OID

    This route can be used to get a direct access to a given content,
    just by submitting an URL like /+/{oid}, where {oid} is the "short"
    sequence OID.
    """
    oid = request.matchdict.get('oid')
    if oid:
        oid = oid.split('::', 1)[0]
        view_name = ''.join(request.matchdict.get('view'))
        sequence = get_utility(ISequentialIntIds)
        reference = sequence.get_full_oid(oid)
        target = get_reference_target(reference)
        if target is not None:
            workflow = IWorkflow(target, None)
            if workflow is not None:
                versions = IWorkflowVersions(target, None)
                if versions is not None:
                    versions = versions.get_versions(workflow.visible_states, sort=True)
                    if versions:
                        target = versions[-1]
        if (target is not None) and not IWorkflowPublicationInfo(target).is_visible(request):
            target = None
        if target is not None:
            if view_name:  # back-office access => last version
                public_url = request.registry.queryMultiAdapter((target, request), IPublicURL,
                                                                name=view_name)
                if public_url is not None:
                    location = public_url.get_url()
                else:
                    location = absolute_url(target, request, view_name)
            else:
                location = canonical_url(target, request, query=request.params)
            if location == request.url:
                # return view response to avoid infinite redirection!
                request.annotations[DISPLAY_CONTEXT] = request.context
                request.context = target
                response = render_view_to_response(target, request, view_name)
            else:
                response = Response()
                response.status_code = 302
                response.location = location
            return response
    raise NotFound()
