#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import Interface

from pyams_content.component.illustration import IIllustration
from pyams_content.features.share.interfaces import ISocialShareInfo
from pyams_content.root import ISiteRoot
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_skin.interfaces.configuration import IConfiguration
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import ContentMeta, PropertyMeta, SchemaMeta
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url, canonical_url


@adapter_config(name='opengraph', context=(ISiteRoot, IPyAMSUserLayer, Interface), provides=IHTMLContentMetas)
class SiteRootOpengraphMetasAdapter(ContextRequestViewAdapter):
    """Opengraph site root metas adapter"""

    order = 15

    def get_metas(self):
        context = self.context
        request = self.request
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language

        configuration = IConfiguration(context)
        title = configuration.title
        description = configuration.description

        # main properties
        yield PropertyMeta('og:type', 'website')
        if title:
            yield PropertyMeta('og:title', title)
        if description:
            yield PropertyMeta('og:description', description)

        # URL and site name
        yield PropertyMeta('og:url', canonical_url(context, request))
        yield PropertyMeta('og:site_name', title)

        # illustration properties
        registry = request.registry
        thumbnail = None
        alt = None
        illustration = registry.queryAdapter(context, IIllustration, name='link')
        if (illustration is None) or (not illustration.has_data()):
            illustration = registry.queryAdapter(context, IIllustration)
        if (illustration is not None) and illustration.has_data():
            data = II18n(illustration).query_attribute('data', lang=lang, request=request)
            if data:
                thumbnail = IThumbnails(data).get_thumbnail('800x600')
                yield PropertyMeta('og:image', absolute_url(thumbnail, request))
                if request.scheme == 'https':
                    yield PropertyMeta('og:image:secure_url', absolute_url(thumbnail, request))
                else:
                    yield PropertyMeta('og:image:url', absolute_url(thumbnail, request))
                yield PropertyMeta('og:image:type', thumbnail.content_type)
                image_size = thumbnail.image_size
                yield PropertyMeta('og:image:width', image_size[0])
                yield PropertyMeta('og:image:height', image_size[1])
                alt = II18n(illustration).query_attribute('alt_title', lang=lang, request=request)
                if alt:
                    yield PropertyMeta('og:image:alt', alt)

        # locales properties
        yield PropertyMeta('og:locale', lang)
        for other_lang in negotiator.offered_languages or ():
            if other_lang != lang:
                yield PropertyMeta('og:locale:alternate', other_lang)

        # twitter properties
        share_info = ISocialShareInfo(request.root, None)
        if share_info is not None:
            if share_info.fb_app_id:
                yield PropertyMeta('fb:app_id', share_info.fb_app_id)
            if share_info.twitter_account:
                yield ContentMeta('twitter:site', share_info.twitter_account)
                yield ContentMeta('twitter:creator', share_info.twitter_creator_account or share_info.twitter_account)
        if title:
            yield ContentMeta('twitter:title', title)
        if description:
            yield ContentMeta('twitter:description', description)
        if thumbnail is not None:
            yield ContentMeta('twitter:card', 'summary_large_image')
            yield ContentMeta('twitter:image', absolute_url(thumbnail, request))
            if alt:
                yield ContentMeta('twitter:image:alt', alt)
        else:
            yield ContentMeta('twitter:card', 'summary')

        # Schema.org properties
        if title:
            yield SchemaMeta('name', title)
        if description:
            yield SchemaMeta('description', description)
        if thumbnail is not None:
            yield SchemaMeta('image', absolute_url(thumbnail, request))
