#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from collections import OrderedDict

from zope.contentprovider.interfaces import IContentProvider
from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, Int
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_i18n.schema import I18nTextLineField


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


class ISearchResultTitle(Interface):
    """Search result title interface"""


class ISearchResultHeader(Interface):
    """Search result header interface"""


class ISearchResultTarget(Interface):
    """Search result target URL interface"""


class ISearchResultRenderer(IContentProvider):
    """Search result renderer interface"""

    url = Attribute("Search result URL")
    title = Attribute("Search result title")
    header = Attribute("Search result header")


FULL_HEADER_DISPLAY = 'full'
START_HEADER_DISPLAY = 'start'
HIDDEN_HEADER_DISPLAY = 'none'

PANELS_HEADER_DISPLAY_MODES = OrderedDict((
    (FULL_HEADER_DISPLAY, _("Display full header")),
    (START_HEADER_DISPLAY, _("Display only header start")),
    (HIDDEN_HEADER_DISPLAY, _("Hide header"))
), )

PANELS_HEADER_DISPLAY_MODES_VOCABULARY = SimpleVocabulary(
    [SimpleTerm(k, title=v) for k, v in PANELS_HEADER_DISPLAY_MODES.items()])


class ISearchResultsPortletRendererBaseSettings(Interface):
    """Search results portlet renderer base settings interface"""

    display_if_empty = Bool(title=_("Display if empty?"),
                            description=_("If 'no', and if no result is found, the portlet "
                                          "will not display anything"),
                            required=True,
                            default=True)

    display_results_count = Bool(title=_("Display results count?"),
                                 description=_("If 'no', results count will not be displayed"),
                                 required=True,
                                 default=True)

    allow_sorting = Bool(title=_("Allow results sorting?"),
                         description=_("If 'no', results will not be sortable"),
                         required=True,
                         default=True)
    
    allow_pagination = Bool(title=_("Allow pagination?"),
                            description=_("If 'no', results will not be paginated"),
                            required=True,
                            default=True)
    
    display_pic_margins = Bool(title=_("Display picture margins?"),
                               description=_("If 'no', pictures will appear with no margins on their sides; "
                                             "WARNING: this is not supported by all graphic themes!"),
                               required=True,
                               default=False)

    display_tags = Bool(title=_("Display tags?"),
                            description=_("If 'no', tags will be hidden"),
                            required=True,
                            default=True)

    display_date = Bool(title=_("Display publication date?"),
                            description=_("If 'no', publication date will be hidden"),
                            required=True,
                            default=False)

    header_display_mode = Choice(title=_("Header display mode"),
                                 description=_("Defines how results headers will be rendered"),
                                 required=True,
                                 vocabulary=PANELS_HEADER_DISPLAY_MODES_VOCABULARY,
                                 default=FULL_HEADER_DISPLAY)
    
    start_length = Int(title=_("Start length"),
                       description=_("If you choose to display only header start, you can "
                                     "specify maximum text length"),
                       required=True,
                       default=120)


class ISearchResultsPortletDefaultRendererSettings(ISearchResultsPortletRendererBaseSettings):
    """Search results portlet default renderer settings interface"""


class ISearchResultsPortletPanelsRendererSettings(ISearchResultsPortletRendererBaseSettings):
    """Search results portlet panels renderer settings interface"""

    button_title = I18nTextLineField(title=_("Button's title"),
                                     description=_("Navigation button's title is normally "
                                                   "defined based on target's content type; you "
                                                   "can override this label by giving a custom "
                                                   "title here"),
                                     required=False)
