#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface, implementer
from zope.location import ILocation
from zope.schema.fieldproperty import FieldProperty

from pyams_content.features.search import ISearchFolder
from pyams_content.features.search.portlet import ISearchResultsPortletSettings
from pyams_content.shared.common import IWfSharedContent
from pyams_default_theme.features.search.portlet.interfaces import ISearchResultHeader, \
    ISearchResultRenderer, ISearchResultTarget, ISearchResultTitle, \
    ISearchResultsPortletDefaultRendererSettings, ISearchResultsPortletPanelsRendererSettings, \
    ISearchResultsPortletRendererBaseSettings, HIDDEN_HEADER_DISPLAY, START_HEADER_DISPLAY, \
    FULL_HEADER_DISPLAY
from pyams_default_theme.interfaces import ISearchResultsPanelView, ISearchResultsView
from pyams_default_theme.shared.common.interfaces import ISharedContentHeadViewletManager
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_skin.interfaces.viewlet import IBreadcrumbs
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import NullAdapter, adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.text import get_text_start
from pyams_utils.url import canonical_url, relative_url
from pyams_viewlet.viewlet import ViewContentProvider, Viewlet, viewlet_config


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


#
# Search folder custom head specificities renderer
#

@viewlet_config(name='search-folder-head', context=ISearchFolder, layer=IPyAMSUserLayer,
                view=Interface, manager=ISharedContentHeadViewletManager, weight=1)
@template_config(template='templates/folder-head-specificities.pt', layer=IPyAMSUserLayer)
class SearchFolderHeadViewlet(Viewlet):
    """Search folder head specificities viewlet"""


#
# Search results portlet base settings and renderers
#

@implementer(ISearchResultsPortletRendererBaseSettings)
class SearchResultsPortletRendererBaseSettings(Persistent, Contained):
    """Search results portlet base renderer settings"""

    display_if_empty = FieldProperty(
        ISearchResultsPortletRendererBaseSettings['display_if_empty'])
    
    display_results_count = FieldProperty(
        ISearchResultsPortletRendererBaseSettings['display_results_count'])

    allow_sorting = FieldProperty(ISearchResultsPortletRendererBaseSettings['allow_sorting'])
    
    allow_pagination = FieldProperty(ISearchResultsPortletRendererBaseSettings['allow_pagination'])
    
    display_pic_margins = FieldProperty(
        ISearchResultsPortletRendererBaseSettings['display_pic_margins'])
    
    display_tags = FieldProperty(
            ISearchResultsPortletRendererBaseSettings['display_tags'])
    
    display_date = FieldProperty(
            ISearchResultsPortletRendererBaseSettings['display_date'])
    
    header_display_mode = FieldProperty(
        ISearchResultsPortletRendererBaseSettings['header_display_mode'])
    
    start_length = FieldProperty(ISearchResultsPortletRendererBaseSettings['start_length'])


class SearchResultsPortletBaseRenderer(PortletRenderer):
    """Search results portlet base renderer"""

    default_page_length = 20
    current_page_length = None

    def update(self):
        settings = self.renderer_settings
        if not settings.allow_pagination:
            self.current_page_length = '999'
        elif 'length' in self.request.params:
            self.current_page_length = str(self.request.params.get('length'))
        else:
            self.current_page_length = str(self.default_page_length)
        super().update()

    def render_item(self, item, **kwargs):
        renderer = self.request.registry.queryMultiAdapter((item, self.request, self),
                                                           ISearchResultRenderer)
        if renderer is not None:
            settings = kwargs.pop('renderer_settings', None)
            if settings is not None:
                renderer.settings = settings
            renderer.update()
            return renderer.render()
        return ''


#
# Search results portlet default renderer
#

@factory_config(provided=ISearchResultsPortletDefaultRendererSettings)
class SearchResultsPortletDefaultRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Search results portlet default renderer settings"""


@adapter_config(context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-results.pt', layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class SearchResultsPortletDefaultRenderer(SearchResultsPortletBaseRenderer):
    """Search results portlet default renderer"""

    label = _("Default search results")

    settings_interface = ISearchResultsPortletDefaultRendererSettings


#
# Search results portlet panels renderer
#

@factory_config(provided=ISearchResultsPortletPanelsRendererSettings)
class SearchResultsPortletPanelsRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Search results portlet panel renderer settings"""

    button_title = FieldProperty(ISearchResultsPortletPanelsRendererSettings['button_title'])


@adapter_config(name='panels',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/search-panels.pt', layer=IPyAMSLayer)
@implementer(ISearchResultsPanelView)
class SearchResultsPortletPanelsRenderer(SearchResultsPortletBaseRenderer):
    """Search results portlet panels renderer"""

    label = _("Paneled search results")

    settings_interface = ISearchResultsPortletPanelsRendererSettings
    default_page_length = 9

    weight = 20


#
# Search results adapters
#

@adapter_config(context=(ILocation, IPyAMSUserLayer, ISearchResultsView), provides=IBreadcrumbs)
class BreadcrumbsAdapter(NullAdapter):
    """Disable breadcrumbs in search results view"""


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, ISearchResultsView),
                provides=ISearchResultTitle)
def shared_content_result_title_adapter(context, request, view):
    """Shared content result title adapter"""
    return II18n(context).query_attribute('title', request=request)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, ISearchResultsView),
                provides=ISearchResultHeader)
def shared_content_result_header_adapter(context, request, view):
    """Shared content result header adapter"""
    return II18n(context).query_attribute('header', request=request)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, ISearchResultsView),
                provides=ISearchResultTarget)
def shared_content_result_target_adapter(context, request, view):
    """Shared content result target URL adapter"""
    if view.settings.force_canonical_url:
        return canonical_url(context, request)
    return relative_url(context, request)


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, ISearchResultsView),
                provides=ISearchResultRenderer)
@template_config(template='templates/search-result.pt', layer=IPyAMSUserLayer)
@implementer(ISearchResultsView)
class WfSharedContentSearchResultRenderer(ViewContentProvider):
    """Shared content search result renderer"""

    @property
    def url(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTarget)

    @property
    def title(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTitle)

    @property
    def header(self):
        display_mode = FULL_HEADER_DISPLAY
        settings = self.view.renderer_settings
        if ISearchResultsPortletRendererBaseSettings.providedBy(settings):
            display_mode = settings.header_display_mode
        if display_mode == HIDDEN_HEADER_DISPLAY:
            return ''
        header = self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                         ISearchResultHeader)
        if display_mode == START_HEADER_DISPLAY:
            header = get_text_start(header, settings.start_length)
        return header


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, ISearchResultsPanelView),
                provides=ISearchResultRenderer)
@template_config(template='templates/search-panel.pt', layer=IPyAMSUserLayer)
class WfSharedContentSearchResultPanelRenderer(WfSharedContentSearchResultRenderer):
    """Shared content search result panel renderer"""
