#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.exceptions import NotFound
from pyramid.view import view_config
from zope.interface import Interface

from pyams_content.features.search.interfaces import ISearchManagerInfo
from pyams_content.root import ISiteRoot
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import LinkMeta
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.url import absolute_url


@adapter_config(name='opensearch',
                context=(Interface, IPyAMSUserLayer, Interface), provides=IHTMLContentMetas)
class OpenSearchMetasAdapter(ContextRequestViewAdapter):
    """OpenSearch metas adapter"""

    def __new__(cls, context, request, view):
        search_info = ISearchManagerInfo(request.root, None)
        if (search_info is None) or (search_info.search_target is None):
            return None
        return ContextRequestViewAdapter.__new__(cls)

    order = 25

    def get_metas(self):
        search_info = ISearchManagerInfo(self.request.root, None)
        yield LinkMeta('search', 'application/opensearchdescription+xml',
                       href=absolute_url(self.request.root, self.request, 'opensearch.xml'),
                       title=II18n(search_info).query_attribute('name', request=self.request))


@view_config(name='opensearch.xml', context=ISiteRoot, request_type=IPyAMSUserLayer,
             renderer='templates/opensearch.pt')
class OpenSearchDescriptionView(object):
    """Site root Opensearch description"""

    def __init__(self, request):
        self.request = request
        self.context = request.context

    def __call__(self):
        self.request.response.content_type = 'application/opensearchdescription+xml'
        return {}

    @property
    def search_info(self):
        search_info = ISearchManagerInfo(self.context, None)
        if (search_info is None) or (search_info.search_target is None):
            raise NotFound()
        return search_info
