#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_content.features.search.interfaces import ISearchFolder
from pyams_content.shared.site import ISiteElementNavigation
from pyams_default_theme import _
from pyams_default_theme.interfaces import IContentHeader, IContentNavigationTitle, \
    IContentSummaryInfo
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.viewlet import IBreadcrumbItem
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.viewlet.breadcrumb import BreadcrumbItem
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.url import absolute_url
from pyams_workflow.interfaces import IWorkflowPublicationInfo


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=ICanonicalURL)
class SearchFolderCanonicalUrlAdapter(ContextRequestAdapter):
    """Search folder canonical URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        return absolute_url(self.context, self.request, view_name, query)


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=IRelativeURL)
class SearchFolderRelativeUrlAdapter(ContextRequestAdapter):
    """Search folder relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        return absolute_url(self.context, self.request, view_name, query)


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=ISiteElementNavigation)
class SearchFolderNavigationAdapter(ContextRequestAdapter):
    """Search folder navigation adapter"""

    @property
    def visible(self):
        if not self.context.visible_in_list:
            return False
        return IWorkflowPublicationInfo(self.context).is_visible(self.request)


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=IContentNavigationTitle)
def search_folder_navigation_title(context, request):
    """Search folder navigation title adapter"""
    i18n = II18n(context)
    title = i18n.query_attribute('navigation_title', request=request)
    if title is None:
        title = i18n.query_attribute('title', request=request)
    return title


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=IBreadcrumbItem)
class SearchFolderBreadcumbAdapter(BreadcrumbItem):
    """Search folder breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('short_name', request=self.request)


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=IContentHeader)
def search_folder_header(context, request):
    """Search folder header adapter"""
    return II18n(context).query_attribute('header', request=request)


@adapter_config(context=(ISearchFolder, IPyAMSUserLayer),
                provides=IContentSummaryInfo)
class SearchFolderSummaryAdapter(ContextRequestAdapter):
    """Search folder summary adapter"""

    @property
    def title(self):
        i18n = II18n(self.context)
        return i18n.query_attribute('navigation_title', request=self.request) or \
               i18n.query_attribute('title', request=self.request)

    @property
    def header(self):
        try:
            return II18n(self.context).query_attribute('header', request=self.request)
        except AttributeError:
            return None

    button_title = _("Consult folder")
