#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import logging

from pyramid.decorator import reify
from pyramid.location import lineage
from zope.component import ComponentLookupError
from zope.interface import implementer

from pyams_cache.beaker import get_cache
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.features.header.interfaces import IHeaderRenderer, IHeaderRendererSettings, \
    IHeaderSettings, IHeaderTarget
from pyams_content.features.renderer.interfaces import HIDDEN_RENDERER_NAME
from pyams_content.root import ISiteRoot
from pyams_default_theme.features.header import SimpleHeaderRendererSettings
from pyams_default_theme.features.header.interfaces import ISimpleHeaderRendererSettings, \
    PYAMS_HEADER_MAIN_CLASS_KEY, PYAMS_HEADER_NAVIGATION_INNER_CLASS_KEY, \
    PYAMS_HEADER_NAVIGATION_MAIN_CLASS_KEY, PYAMS_HEADER_TEMPLATE_NAME_KEY, \
    PYAMS_HEADER_TITLE_INNER_CLASS_KEY, \
    PYAMS_HEADER_TITLE_MAIN_CLASS_KEY
from pyams_default_theme.features.header.skin.interfaces import IHeaderClass, \
    IInnerNavigationClass, IInnerTitleClass, INavigationClass, ITitleClass
from pyams_default_theme.features.renderer import BaseContentRenderer
from pyams_portal.interfaces import PREVIEW_MODE
from pyams_portal.portlet import PORTLETS_CACHE_KEY, PORTLETS_CACHE_NAME, PORTLETS_CACHE_REGION
from pyams_sequence.interfaces import IInternalReference
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import get_view_template, template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'

from pyams_default_theme import _


logger = logging.getLogger('PyAMS (content)')


#
# Base header renderer
#

@implementer(IHeaderRenderer)
class BaseHeaderRenderer(BaseContentRenderer):
    """Base header renderer"""

    name = None
    settings_key = None

    @reify
    def settings_target(self):
        context = self.request.display_context
        if context is None:
            context = self.context
        return get_parent(context, IHeaderTarget)

    @reify
    def settings(self):
        if self.settings_interface is None:
            return None
        target = self.settings_target
        settings = IHeaderSettings(target)
        while settings.inherit:
            settings = IHeaderSettings(settings.parent)
        return settings.settings

    @reify
    def cache_key(self):
        return PORTLETS_CACHE_KEY.format(portlet=self.name,
                                         hostname=self.request.host,
                                         context=ICacheKeyValue(self.settings_target),
                                         lang=self.request.locale_name)

    @property
    def main_header_class(self):
        request = self.request
        header_class = request.annotations.get(PYAMS_HEADER_MAIN_CLASS_KEY)
        if header_class is None:
            header_class = request.registry.queryMultiAdapter((request.context, request),
                                                              IHeaderClass, default='')
        return header_class

    @property
    def main_title_class(self):
        request = self.request
        title_class = request.annotations.get(PYAMS_HEADER_TITLE_MAIN_CLASS_KEY)
        if title_class is None:
            title_class = request.registry.queryMultiAdapter((request.context, request),
                                                             ITitleClass, default='')
        return title_class

    @property
    def inner_title_class(self):
        request = self.request
        title_class = request.annotations.get(PYAMS_HEADER_TITLE_INNER_CLASS_KEY)
        if title_class is None:
            title_class = request.registry.queryMultiAdapter((request.context, request),
                                                             IInnerTitleClass, default='')
        return title_class

    @property
    def main_navigation_class(self):
        request = self.request
        nav_class = request.annotations.get(PYAMS_HEADER_NAVIGATION_MAIN_CLASS_KEY)
        if nav_class is None:
            nav_class = request.registry.queryMultiAdapter((request.context, request),
                                                           INavigationClass, default='')
        return nav_class

    @property
    def inner_navigation_class(self):
        request = self.request
        nav_class = request.annotations.get(PYAMS_HEADER_NAVIGATION_INNER_CLASS_KEY)
        if nav_class is None:
            nav_class = request.registry.queryMultiAdapter((request.context, request),
                                                           IInnerNavigationClass, default='')
        return nav_class

    @property
    def search_target(self):
        return get_reference_target(self.settings.search_target, request=self.request)

    @property
    def template_name(self):
        return self.request.annotations.get(PYAMS_HEADER_TEMPLATE_NAME_KEY, '')

    def get_template(self, template_name=''):
        if template_name:
            template = get_view_template(name=template_name)
        else:
            template = get_view_template(name=self.template_name)
        try:
            return template(self)
        except ComponentLookupError:
            template = get_view_template()
            return template(self)

    def render(self, template_name=''):
        request = self.request
        context = request.context
        # root condition applies only when not displaying a shared content
        if ISiteRoot.providedBy(context) and \
                not self.settings.can_apply_on_root:
            return ''
        # don't use cache in preview mode
        preview_mode = request.annotations.get(PREVIEW_MODE, False)
        if preview_mode:
            return self.get_template(template_name)
        portlets_cache = get_cache(PORTLETS_CACHE_REGION, PORTLETS_CACHE_NAME)
        cache_key = self.cache_key
        if context is not self.context:  # display shared content
            cache_key = '{0}::shared'.format(cache_key)
        if template_name:
            cache_key = '{}::{}'.format(cache_key, template_name)
        try:
            result = portlets_cache.get_value(cache_key)
            logger.debug("Retrieving header content from cache key {}".format(cache_key))
        except KeyError:
            result = self.get_template(template_name)
            portlets_cache.set_value(cache_key, result)
            logger.debug("Storing header content to cache key {}".format(cache_key))
        return result


#
# Hidden header renderer
#

@adapter_config(name=HIDDEN_RENDERER_NAME, context=(IHeaderTarget, IPyAMSLayer),
                provides=IHeaderRenderer)
class HiddenHeaderRenderer(BaseHeaderRenderer):
    """Hidden header renderer"""

    name = HIDDEN_RENDERER_NAME
    label = _("Hidden header")
    weight = -999

    def render(self, template_name=''):
        return ''


#
# Simple header renderer
#

SIMPLE_HEADER_RENDERER_NAME = 'PyAMS simple header'


@adapter_config(name=SIMPLE_HEADER_RENDERER_NAME, context=(IHeaderTarget, IPyAMSLayer),
                provides=IHeaderRenderer)
@adapter_config(name=SIMPLE_HEADER_RENDERER_NAME, context=(IHeaderSettings, IPyAMSLayer),
                provides=IHeaderRenderer)
@template_config(template='templates/simple-header.pt', layer=IPyAMSLayer)
class SimpleHeaderRenderer(BaseHeaderRenderer):
    """Simple header renderer"""

    name = SIMPLE_HEADER_RENDERER_NAME
    label = _("PyAMS simple header with banner and tabs")
    weight = 1

    settings_key = 'PyAMS::simple'
    settings_interface = ISimpleHeaderRendererSettings

    def update(self):
        super(SimpleHeaderRenderer, self).update()
        self.request.annotations['REQUEST_PATH'] = list(lineage(self.request.context))

    @staticmethod
    def get_link_info(link):
        return IAssociationInfo(link)

    def is_active(self, link):
        if IInternalReference.providedBy(link):
            target = link.get_target()
            if target is not None:
                return target in self.request.annotations['REQUEST_PATH']
        return False


@adapter_config(context=SimpleHeaderRenderer, provides=IHeaderRendererSettings)
def simple_header_renderer_settings_factory(context):
    """Simple header renderer settings factory"""
    return SimpleHeaderRendererSettings()
