#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.interface import implementer
from zope.location import Location
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.interfaces import ASSOCIATION_CONTAINER_KEY
from pyams_content.features.header.interfaces import IHeaderSettings, IHeaderTarget
from pyams_content.features.menu.interfaces import IMenusContainer, IMenusContainerTarget
from pyams_default_theme.features.header.interfaces import ISimpleHeaderRendererSettings, ISimpleHeaderMenusContainer
from pyams_default_theme.layer import IPyAMSDefaultLayer
from pyams_file.property import FileProperty
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import ViewContentProvider, contentprovider_config


@contentprovider_config(name='pyams.header', layer=IPyAMSDefaultLayer)
class HeaderContentProvider(ViewContentProvider):
    """Header content provider"""

    header = None
    renderer = None

    def update(self):
        parent = get_parent(self.context, IHeaderTarget)
        if parent is not None:
            header = self.header = IHeaderSettings(parent, None)
            if header is not None:
                renderer = self.renderer = header.get_renderer(self.request)
                if renderer is not None:
                    renderer.update()

    def render(self, template_name=''):
        if (self.header is None) or (self.renderer is None):
            return ''
        return self.renderer.render(template_name)


#
# Simple header renderer settings
#

SIMPLE_HEADER_MENUS_KEY = '{0}::menus'.format(ASSOCIATION_CONTAINER_KEY)


@implementer(ISimpleHeaderRendererSettings, IMenusContainerTarget)
class SimpleHeaderRendererSettings(Persistent, Location):
    """Simple header renderer settings"""

    apply_on_root = FieldProperty(ISimpleHeaderRendererSettings['apply_on_root'])
    banner = FileProperty(ISimpleHeaderRendererSettings['banner'])
    logo = FileProperty(ISimpleHeaderRendererSettings['logo'])
    search_target = FieldProperty(ISimpleHeaderRendererSettings['search_target'])

    @property
    def can_apply_on_root(self):
        return self.apply_on_root

    @property
    def menus(self):
        return get_annotation_adapter(self, SIMPLE_HEADER_MENUS_KEY, IMenusContainer,
                                      markers=ISimpleHeaderMenusContainer, name='++ass++menus')


@adapter_config(name='menus', context=ISimpleHeaderRendererSettings, provides=IMenusContainer)
def simple_header_menus_adapter(context):
    """Simple header menus adapter"""
    return context.menus


@adapter_config(name='menus', context=ISimpleHeaderRendererSettings, provides=ISublocations)
class SimpleHeaderRendererSettingsSublocations(ContextAdapter):
    """Simple header renderer settings sub-locations adapter"""

    def sublocations(self):
        return self.context.menus.values()
